/**
 * A helper class used by the Sight Layer to represent a source of vision or illumination.
 * @param {PlaceableObject} object      The object responsible for the PointSource
 * @abstract
 */
class PointSource {
  constructor(object) {

    /**
     * The PlaceableObject which is the origin of this PointSource.
     * @type {PlaceableObject}
     */
    this.object = object;
  }

  /* -------------------------------------------- */

  /**
   * The type of source represented by this data structure.
   * Each subclass must implement this attribute.
   * @type {string}
   */
  static sourceType;

  /* -------------------------------------------- */

  /**
   * A flag for whether this source is currently rendered or not.
   * @type {boolean}
   */
  active = false;

  /**
   * @typedef {Object} PointSourceAnimationConfiguration
   * @property {string} [label]                                   The human-readable (localized) label for the animation
   * @property {Function} [animation]                             The animation function that runs every frame
   * @property {AdaptiveIlluminationShader} [illuminationShader]  A custom illumination shader used by this animation
   * @property {AdaptiveColorationShader} [colorationShader]      A custom coloration shader used by this animation
   * @property {AdaptiveBackgroundShader} [backgroundShader]      A custom background shader used by this animation
   * @property {number} [seed]                                    The animation seed
   * @property {number} [time]                                    The animation time
   */

  /**
   * The animation configuration applied to this source
   * @type {PointSourceAnimationConfiguration}
   */
  animation = {};

  /**
   * The object of data which configures how this source is rendered
   * @type {object}
   */
  data = {};

  /**
   * The maximum radius of emission for this source
   * @type {number}
   */
  radius = 0;

  /**
   * The restricted line-of-sight polygon that is generated by the origin and radius of this source.
   * @type {PointSourcePolygon}
   */
  los;

  /**
   * PIXI Geometry generated to draw meshes.
   * @type {PIXI.Geometry|null}
   * @private
   */
  _sourceGeometry = null;

  /**
   * A Graphics object with pre-computed geometry used for masking based on line-of-sight.
   * @type {PIXI.Graphics}
   */
  losMask = new PIXI.LegacyGraphics();

  /**
   * Additional information which controls whether certain behaviors of the source must be enforced
   * @type {Object<string,boolean|number>}
   * @protected
   */
  _flags = {};

  /**
   * To track meshes initialization
   * @type {boolean}
   * @protected
   */
  _meshesInit = false;

  /**
   * The offset in pixels applied to create soft edges.
   * @type {number}
   */
  static EDGE_OFFSET = -8;

  /* -------------------------------------------- */

  /**
   * The x-coordinate of the point source origin.
   * @type {number}
   */
  get x() {
    return this.data.x;
  }

  /**
   * The y-coordinate of the point source origin.
   * @type {number}
   */
  get y() {
    return this.data.y;
  }

  /**
   * The type of source represented by this data structure.
   * @type {string}
   */
  get sourceType() {
    return this.constructor.sourceType;
  }

  /**
   * The elevation of the object bound to this base source, if any.
   * Returns the canvas primary background elevation otherwise.
   * @type {number}
   */
  get elevation() {
    return this.object?.document?.elevation ?? canvas.primary.background.elevation;
  }

  /**
   * If the source is animated or not.
   * @type {boolean}
   */
  get isAnimated() {
    const {animation} = this.animation;
    return !(!animation || !this._meshesInit || (this.radius === 0));
  }

  /* -------------------------------------------- */
  /*  Point Source Methods                        */
  /* -------------------------------------------- */

  /**
   * A point is contained with the area of the source if it is within both the FOV circle and the LOS polygon.
   * @param {Point} point   The point to test
   * @returns {boolean}     Is the point contained
   */
  containsPoint(point) {
    return this.los.contains(point.x, point.y);
  }

  /* -------------------------------------------- */

  /**
   * Steps that must be performed when the base source is destroyed.
   */
  destroy() {
    this._meshesInit = false;
    this._sourceGeometry?.destroy();
  }

  /* -------------------------------------------- */

  /** @ignore */
  initialize(data={}) {
    throw new Error("Each subclass of PointSource must implement the initialize method");
  }

  /* -------------------------------------------- */

  /**
   * Refresh the state and uniforms of the BaseSource
   * @abstract
   */
  refreshSource() {
    throw new Error("A subclass of PointSource must implement the refreshSource method.");
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /**
   * Create or update the source geometry with a polygon shape
   * Triangulate the form and create buffers
   * @param {PIXI.Polygon} polygon   The pixi polygon
   * @protected
   */
  _updateLosGeometry(polygon) {
    const polyMesher = new PolygonMesher(polygon, {
      normalize: true,
      x: this.x,
      y: this.y,
      radius: this.radius,
      offset: this._flags.renderSoftEdges ? PointSource.EDGE_OFFSET : 0
    });

    this._sourceGeometry = polyMesher.triangulate(this._sourceGeometry);
  }

  /* -------------------------------------------- */

  /**
   * Configure the parameters of the polygon that is generated for this source.
   * @returns {PointSourcePolygonConfig}
   * @abstract
   * @protected
   */
  _getPolygonConfiguration() {}

  /* -------------------------------------------- */

  /**
   * Create the LOS polygon for this Light Source instance using provided parameters.
   * @returns {PointSourcePolygon|PIXI.Polygon}
   * @abstract
   * @protected
   */
  _createPolygon() {
    const origin = {x: this.data.x, y: this.data.y};
    return CONFIG.Canvas.losBackend.create(origin, this._getPolygonConfiguration());
  }

  /* -------------------------------------------- */

  /**
   * Create or update the source geometry and create meshes if necessary
   * @param {PIXI.Polygon} polygon    A pixi polygon
   * @protected
   */
  _initializeMeshes(polygon) {
    this._updateLosGeometry(polygon);
    if ( this._sourceGeometry && !this._meshesInit ) {
      this._createMeshes();
      this._meshesInit = true;
    }
  }

  /* -------------------------------------------- */

  /**
   * Create a new Mesh for this source using a provided shader class
   * @param {Function} shaderCls  The subclass of AdaptiveLightingShader being used for this Mesh
   * @returns {PIXI.Mesh}         The created Mesh
   * @protected
   */
  _createMesh(shaderCls) {
    const state = new PIXI.State();
    const mesh = new PointSourceMesh(this._sourceGeometry, shaderCls.create(), state);
    mesh.drawMode = PIXI.DRAW_MODES.TRIANGLES;
    Object.defineProperty(mesh, "uniforms", {get: () => mesh.shader.uniforms});
    return mesh;
  }

  /* -------------------------------------------- */

  /**
   * Create all meshes needed with this PointSource
   * @abstract
   */
  _createMeshes() {
    throw new Error("A subclass of PointSource must implement the _createMeshes method.");
  }

  /* -------------------------------------------- */

  /**
   * Update the position and size of the mesh each time it is drawn.
   * @param {PIXI.Mesh} mesh      The Mesh being updated
   * @returns {PIXI.Mesh}         The updated Mesh
   * @protected
   */
  _updateMesh(mesh) {
    mesh.position.set(this.data.x, this.data.y);
    mesh.scale.set(this.radius);
    mesh.visible = mesh.renderable = true;  // If we have reached this point, the mesh should be visible
    return mesh;
  }

  /* -------------------------------------------- */
  /*  Animation Functions                         */
  /* -------------------------------------------- */

  /**
   * Animate the BaseSource, if an animation is enabled and if it currently has rendered containers.
   * @param {number} dt         Delta time.
   */
  animate(dt) {
    if ( !this.isAnimated ) return;
    const {animation, ...options} = this.animation;
    return animation.call(this, dt, options);
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /**
   * @deprecated since v10, will be removed in v11.
   * @ignore
   */
  getPowerOf2Size() {
    const perf = canvas.performance.textures;
    const maxP2 = canvas.getGLParameter("MAX_RENDERBUFFER_SIZE") >> perf.p2StepsMax;
    const rtP2 = PIXI.utils.nextPow2(this.radius * 2) >> perf.p2Steps;
    return Math.clamped(64, rtP2, maxP2);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v10
   * @ignore
   */
  get limited() {
    const msg = "PointSource#limited is deprecated in favor of PointSourcePolygon#isConstrained.";
    foundry.utils.logCompatibilityWarning(msg, {since: 10, until: 12});
    return this.los.isConstrained;
  }
}
