// *************************************************************************************************************
// * This is the user program file for the OTG HID demonstration package (AT90USB647/1287)
// * In addition to the OTG firmware features, it includes :
// *  - request specific OTG features from pushbuttons :
// *    + HOST MODE :
// *      - HWB : toggle Vbus (from any state of an A-Device)
// *      - Joystick SELECT/CENTER : request HNP (from A-HOST or B-HOST)
// *    + PERIPH MODE :
// *      - HWB : send a SRP (from B-IDLE only)
// *      - Joystick SELECT/CENTER : toggle Host' Hid LED  (from A-PERIPH or B-PERIPH only)
// *
// *  - handle a basic HID mouse application : 
// *     -> in Device mode, the device sends a report indicating a click on CENTER pushbutton
// *     -> in Host mode, a mouse or HID compatible device can be connected. The firmware toggle the LED state on each click
// *
// *  - LED for messaging feature : 4 monocolor LED on STK525, 2 bicolor LED on USBKEY
//*     Refer to "otg_user_task." for LED assignment
// *     -> Led_noresponse (Red) : "Device No Response" signalling
// *     -> Led_usb_connected (Green) : role identifier, this LED is ON in PERIPHERAL mode, when enumerated
// *     -> Led_hid (Green) : on HOST side, this LED is toggled each time the Peripheral sends a CLICK report
// *     -> Led_unsupported (Red) : "Unsupported Device" signalling
// *************************************************************************************************************

// MCU is using an external 8 MHz quartz
// User must indicate the clock prescaler configuration set up (in order to generate constants 2ms interruptions)
#define   OSC_CORE_PRESCALER    NO_PRESC
 
//___________I N C L U D E S___________________________
#include "config.h"
#include "conf_usb.h"
#include "lib_mcu/usb/usb_drv.h"
#include "modules/usb/usb_task.h"
#include "modules/usb/device_chap9/usb_standard_request.h"
#include "modules/usb/host_chap9/usb_host_task.h"
#include "modules/usb/device_chap9/usb_device_task.h"
#include "modules/usb/host_chap9/usb_host_enum.h"
#include "lib_mcu/timer/timer16_drv.h"
#include "lib_mcu/usb/usb_drv.h"
#include "otg_user_task.h"
//#include "lib_mcu/adc/adc_drv.h"
#include "lib_mcu/pll/pll_drv.h" 


//___________V A R I A B L E S__________________________
U8 PIPE_SPIDER_IN;

volatile U8 button_high;

volatile U8 device_connected;
volatile U8 previous_session, previous_vbus;

volatile S8 mouse_move;
volatile S8 Usb_report[4];
volatile U8 Usb_report_status;

#if (USE_TIMER16 == BOTH_TIMER16)
  U8  timer16_selected;
#endif


// ***************************************
// **
// **  MAIN PROGRAM
// **
// ***************************************

void otg_user_task_init(void)
{
  // Board init
  Leds_init();
  Joy_init();
  Hwb_button_init();
  Leds_off();
  
  previous_session = SESSION_DEVICE;  // to make the function Led_print_otg_status() display the first status
  previous_vbus = 0;
  device_connected = SESSION_OFF;
  Led_print_otg_status();

  Enable_interrupt();
  
  Clear_usb_report();
  Usb_report_status = USB_READ_PACKET;
  button_high = FALSE;
}


void otg_user_task(void)
{
U8 i;

  // ==============================
  // ** USB low-level management **
  // ==============================

  // ================
  // In HOST mode, USB high-level management
  // ================
  if (Is_usb_host_enabled())
  {
    if (Is_host_ready())
    {
    // In HOST mode, check if a new peripheral has been connected
      if (Is_new_device_connection_event())
      {
        for(i=0;i<Get_nb_supported_interface();i++)
        {
           if (Get_class(i)==HID_CLASS && Get_protocol(i)==MOUSE_PROTOCOL)
           {
              device_connected = SESSION_HOST;
              Led_usb_connected_off();
              Led_hid_off();
              host_hid_set_idle();
              host_get_hid_repport_descriptor();
              PIPE_SPIDER_IN = host_get_hwd_pipe_nb(Get_ep_addr(i,0));
              Host_select_pipe(PIPE_SPIDER_IN);
              Host_continuous_in_mode();
              Host_unfreeze_pipe();
              button_high = FALSE;
              mouse_move = 0;
              break;
           }
        }
      }
    }
    else
    {
      Led_usb_connected_off();
      Led_hid_off();
      device_connected = SESSION_OFF;
    }
      

    // Disconnection event
    if(Is_device_disconnection_event())
    {
      Led_usb_connected_off();
      Led_hid_off();
      device_connected = SESSION_OFF;
    }

    // In HOST mode, check for new packet received
    if (device_connected == SESSION_HOST)  // Peripheral is connected and enumerated
    {
      // 1. Get new data from Device
      Host_select_pipe(PIPE_SPIDER_IN);
      if ((Is_host_in_received()) && (Is_host_stall()==FALSE) && (Usb_report_status != USB_NEW_PACKET))
      {
        i = 0;
        while ((UPBCLX != 0) && (i != 4))
        {
          Usb_report[i] = Host_read_byte();
          i++;
        }
        Usb_report_status = USB_NEW_PACKET;

        Host_ack_in_received();
        Host_send_in();
      }
    }
  }

  // ================
  // In DEVICE mode, USB high-level management
  // ================
  if (Is_usb_device_enabled())
  {
    if (usb_configuration_nb != 0)  // is device configured and enumerated ?
    {
      Led_usb_connected_on();
      device_connected = SESSION_DEVICE;
      Usb_select_endpoint(EP_SPIDER_IN);    // endpoint sending data to HOST
      if ((Is_usb_write_enabled()) && (Usb_report_status == USB_NEW_PACKET))
      {
        Usb_write_byte(Usb_report[0]);
        Usb_write_byte(Usb_report[1]);
        Usb_write_byte(Usb_report[2]);
        Usb_write_byte(Usb_report[3]);
        Usb_report_status = USB_READ_PACKET;
        Usb_ack_fifocon();               // Send data over the USB
        Clear_usb_report();
      }
    }
    else
    {
      Led_usb_connected_off();
      Led_hid_off();
      device_connected = SESSION_OFF;
    }
  }


  // =======================
  // ** Application level **
  // DEVICE mode : send HID report with information about CENTER pushbutton (push / release)
  // HOST mode : toggle the LED according to button position
  // =======================
  if (device_connected == SESSION_DEVICE)
  {
    // Acquires moves from mouse
    // If notorious change (more than 4%), the difference is sent to Host
    if (Usb_report_status != USB_NEW_PACKET)
    {
      if (Is_joy_select())
      {
        // Pushbutton debounce is handled by USB Pipe polling frequency (EP_INTERVAL_1 value in "usb_descriptors.h")
        Usb_report[0] = 0x01;
        Usb_report_status = USB_NEW_PACKET;
      }
      else
      {
        Usb_report[0] = 0x00;
        Usb_report_status = USB_NEW_PACKET;
      }
    }
  }

  if (device_connected == SESSION_HOST)
  {
    if (Usb_report_status == USB_NEW_PACKET)
    {
      if ((Usb_report[0]&0x01) == 0)
      {
        if (button_high == TRUE)    { button_high = FALSE; }
      }
      else
      {
        if (button_high == FALSE)
        {
          button_high = TRUE;
          Led_hid_toggle();
        }        
      }
      Usb_report_status = USB_READ_PACKET;
      Clear_usb_report();
    }
  }

  
  
  // =================================================================
  // OTG User Request Management
  // According to the Role, pushbuttons will initiate different events
  // =================================================================
  if (Is_usb_device_enabled())
  {
    if (!Is_usb_id_host() && (usb_configuration_nb == 0))
    {
      // SRP Request : HWB Button
      // ***********
      if (Is_hwb())
      {
        Bp_delay_debounce();
        while (Is_hwb());
        Bp_delay_debounce();
        Set_user_request_srp();
      }
    }
    if (Is_usb_id_host())
    {
      // VBUS Toggle Request : HWB Button   // to allow session ends when in A-PERIPHERAL mode
      // *******************
      if (Is_hwb())
      {
        Bp_delay_debounce();
        while (Is_hwb());
        Bp_delay_debounce();
        Set_user_request_vbus();
      }
    }
  }

  if (Is_usb_host_enabled())
  {
    // VBUS Toggle Request : HWB Button
    // *******************
    if (Is_hwb())
    {
      Bp_delay_debounce();
      while (Is_hwb());
      Bp_delay_debounce();
      Set_user_request_vbus();
    }
    
    // HNP Toggle Request : CENTER Button
    // *******************
    if (Is_joy_select())
    {
      Bp_delay_debounce();
      while (Is_joy_select());
      Bp_delay_debounce();
      Set_user_request_hnp();
    }
  }

  Led_print_otg_status();
}





// This function actualizes the OTG status line when called
// It displays the VBUS state and the current SESSION (HOST, DEVICE, OFF ; other states can be added - 4 letters)
void Led_print_otg_status(void)
{
  U8 vbus_state = 0;
  if ((Is_usb_vbus_high()) || ((Is_usb_id_host()) && ((PORTE&0x80) != 0)))    { vbus_state = 1; }
  if ((previous_session != device_connected) || (previous_vbus != vbus_state))
  {
    switch (device_connected)
    {
    case SESSION_HOST:
      Led_usb_connected_off();
      break;

    case SESSION_DEVICE:
      Led_usb_connected_on();
      break;

    case SESSION_OFF:
      Led_usb_connected_off();
      break;

    default:
      Led_usb_connected_off();
      break;
    }
    previous_session = device_connected;
    previous_vbus = vbus_state;
  }
}



//-----------------------------------------------------------------
// MESSAGING FEATURES
// ******************

void Otg_messaging_init(void)
{
  Led_unsupported_off();
  Led_noresponse_off();
}

// Function that get the string number as input and writes the corresponding FAILURE string on LCD
void Otg_output_failure_msg(U8 str_nb)
{
  switch (str_nb)
  {
    case OTGMSG_UNSUPPORTED:
      // Put here handler code for this message
      Led_unsupported_on();
      break;
      
    case OTGMSG_UNSUPPORTED_HUB:
      // Put here handler code for this message
      Led_unsupported_on();
      break;

    case OTGMSG_DEVICE_NO_RESP:
      // Put here handler code for this message
      Led_noresponse_on();
      break;
      
    case OTGMSG_SRP_A_NO_RESP:
      // Put here handler code for this message
      Led_noresponse_on();
      break;
      
    default:
      break;
  }
}

// Clears the line of FAILURE messages
void Otg_clear_failure_message()
{
  Led_unsupported_off();
  Led_noresponse_off();  
}


