/*This file has been prepared for Doxygen automatic documentation generation.*/
//! \file *********************************************************************
//!
//! \brief This file manages the USB Host CDC control application.
//!
//! - Compiler:           IAR EWAVR and GNU GCC for AVR
//! - Supported devices:  AT90USB1287, AT90USB1286, AT90USB647, AT90USB646
//!
//! \author               Atmel Corporation: http://www.atmel.com \n
//!                       Support and FAQ: http://support.atmel.no/
//!
//! ***************************************************************************

/* Copyright (c) 2007, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//_____  I N C L U D E S ___________________________________________________

#include "config.h"
#include "conf_usb.h"
#include "host_cdc_task.h"
#include "modules/usb/host_chap9/usb_host_task.h"
#include "modules/usb/host_chap9/usb_host_enum.h"
#include "lib_mcu/usb/usb_drv.h"
#ifdef CDC_USE_UART
   #include "lib_mcu/uart/uart_lib.h"
#endif

//_____ M A C R O S ________________________________________________________

#ifndef LOG_STR_CODE
#define LOG_STR_CODE(str)
#else
U8 code log_cdc_connect[]="CDC Connected";
#endif

//_____ D E F I N I T I O N S ______________________________________________


//_____ D E C L A R A T I O N S ____________________________________________

U8  cdc_connected;
U8  cdc_interface_comm;
U8  cdc_cpt_sof;

U8  pipe_cdc_comm_int;
U8  pipe_cdc_data_bulkin;
U8  pipe_cdc_data_bulkout;

U8  tx_counter;
U8  rx_counter;
U8  cdc_stream_out_array[CDC_STREAM_OUT_SIZE];
U8  cdc_stream_in_array[CDC_STREAM_IN_SIZE];


//!@brief This function initializes the Host USB CDC application
//!
void host_cdc_task_init(void)
{
#ifdef CDC_USE_UART
   uart_init();
#endif
   Leds_init();
   cdc_connected = 0;
   cdc_interface_comm = 0;
   Joy_init();
}


//! @brief This function manages the HOST CDC application
//!
void host_cdc_task(void)
{
   U8 i;

   if(Is_host_ready())
   {
      if(Is_new_device_connection_event())
      {
         cdc_connected=0;
         Led0_on();
         for(i=0;i<Get_nb_supported_interface();i++)
         {
            // Data Interface
            if((Get_class(i)==CDC_DATA_CLASS) && (Get_protocol(i)==CDC_DATA_PROTOCOL))
            {
               cdc_connected=1;
               Host_enable_sof_interrupt();
               LOG_STR_CODE(log_cdc_connect);
               if(Is_ep_addr_in(Get_ep_addr(i,0)))
               {  // Yes associate it to the CDC Data IN pipe
                  pipe_cdc_data_bulkin = host_get_hwd_pipe_nb(Get_ep_addr(i,0));
                  pipe_cdc_data_bulkout = host_get_hwd_pipe_nb(Get_ep_addr(i,1));
               }
               else
               {  // No, invert...
                  pipe_cdc_data_bulkin = host_get_hwd_pipe_nb(Get_ep_addr(i,1));
                  pipe_cdc_data_bulkout = host_get_hwd_pipe_nb(Get_ep_addr(i,0));
               }
               Host_select_pipe(PIPE_CDC_DATA_IN);
               Host_continuous_in_mode();
               Host_unfreeze_pipe();
               break;
            }
            // Management Interface
            #ifdef   CDC_USE_MANAGEMENT_INTERFACE
            if(Get_class(i)==CDC_COMM_CLASS && Get_protocol(i)==CDC_COMM_PROTOCOL)
            {
               cdc_interface_comm = i;      // store interface number
               pipe_cdc_comm_int = host_get_hwd_pipe_nb(Get_ep_addr(i,0));
               Host_select_pipe(PIPE_CDC_COMM);
               Host_continuous_in_mode();
               Host_unfreeze_pipe();
            }
            #endif
         }
         // Open port (according cdc spec 1.1 chapter 6.2.14)
         usb_request.bmRequestType = USB_SETUP_SET_CLASS_INTER;
         usb_request.bRequest      = SETUP_CDC_SET_CONTROL_LINE_STATE;
         usb_request.wValue        = 0x03;
         usb_request.wIndex        = 0;
         usb_request.wLength       = 0;
         usb_request.uncomplete_read = FALSE;
         if( CONTROL_GOOD == host_send_control(0))
         {
            Led1_on();
         }         
      }

      if(cdc_connected)
      {
         // Check DATA_PIPE_IN for incoming data
         // ************************************
         // 1 -> UART-USB Mode
         //   Data received is automatically sent on the UART
         // 2 -> Normal Mode
         //   Data received is stored in the "cdc_stream_in_array[CDC_STREAM_IN_SIZE]" array and may be read by firmware
         //   When firmware reads the array it must do it entirely and then set "rx_counter" variable to 0
         //   It must not partially read the array and decrement rx_counter (will lead to array overflow)
         Host_select_pipe(PIPE_CDC_DATA_IN);
         if (Is_host_in_received() && (Is_host_stall()==FALSE))
         {
#ifdef CDC_USE_UART
            while (Host_data_length_U8() != 0)
            {
              uart_putchar(Host_read_byte());
            }
            Host_ack_in_received();    // pipe is empty
            Host_send_in();            // ready to receive more data
#else
            while ((rx_counter != CDC_STREAM_IN_SIZE) && (Host_data_length_U8() != 0))
            {
              cdc_stream_in_array[rx_counter] = Host_read_byte();
              rx_counter++;
            }
            if (Host_data_length_U8() == 0)
            {
               Host_ack_in_received();    // pipe is empty
               Host_send_in();            // ready to receive more data
            }
#endif
         }


         // Check if data to send on DATA_PIPE_OUT state
         // ********************************************
         // Data to be sent is stored in the "cdc_stream_out_array[CDC_STREAM_OUT_SIZE]" array
         // Data may come indifferently from neither UART or RAM (firmware)
         // Data is sent to CDC Device when :
         //    - user requires it, by calling "cdc_pipe_out_usb_flush()" function
         //    - time-out period is elapsed (CDC_NB_MS_BEFORE_FLUSH = number of SOF seen)
         //    - buffer is full (tx_counter = CDC_STREAM_OUT_SIZE)

#ifdef CDC_USE_UART
         // Check if new byte in USART, to be stored for USB
         if (uart_test_hit() && (tx_counter != CDC_STREAM_OUT_SIZE))
         {
           cdc_stream_out_array[tx_counter] = uart_getchar();
           tx_counter++;
         }
#endif

         // Check if pipe flush is needed (buffer full or time-out period elapsed)
         if(((cdc_cpt_sof>=CDC_NB_MS_BEFORE_FLUSH) && (tx_counter!=0)) || (tx_counter == CDC_STREAM_OUT_SIZE))  //Flush buffer by Timeout
         {
            cdc_cpt_sof=0;
            cdc_pipe_out_usb_flush();
         }


         // Check in COMM_PIPE_IN for incoming notifications
         // ************************************************
         Host_select_pipe(PIPE_CDC_COMM);
         if (Is_host_in_received())
         {
            // Handle here notification messages sent by device
            // Notifications messages have the following structure :
            //  bmRequestType - bNotification - wValue - wIndex - wLength - Data     (wLength is the number of bytes of the Data field)

            //   - NETWORK_CONNECTION : indicates that device has connected to network
            //   - RESPONSE_AVAILABLE : indicates that device has a ready encapsulated response (wait for host request)
            //   - SERIAL_STATE : indicates state of device' UART (errors, carriers and misc. signals)
            //   - etc...

            // ...and now...just coding...
            Host_ack_in_received();
            Host_send_in();
         }
      }
   }

   // Device disconnection...
   if(Is_device_disconnection_event())
   {
      Leds_off();
      cdc_connected=0;
      cdc_interface_comm = 0;
   }
}


//! This function increments the cdc_cpt_sof counter each times
//! the USB Start Of Frame interrupt subroutine is executed (1ms)
//! Usefull to manage time delays
//!
void sof_action(void)
{
   cdc_cpt_sof++;
}


//! This function sends the data stored in the temporary transmit buffer.
//! This function does nothing if there is no data in the buffer.
//!
void cdc_pipe_out_usb_flush (void)
{
   Host_select_pipe(PIPE_CDC_DATA_IN);    // BULK IN must be frozen else BULK OUT may not be sent
   Host_freeze_pipe();
   if (PIPE_GOOD == host_send_data(PIPE_CDC_DATA_OUT, tx_counter, cdc_stream_out_array))
   {
      tx_counter = 0;                     // if frame not sent, will try again next time (no data loss)
   }
   Host_select_pipe(PIPE_CDC_DATA_IN);
   Host_unfreeze_pipe();
}


