/*This file has been prepared for Doxygen automatic documentation generation.*/
//! \file *********************************************************************
//!
//! \brief This file manages the audio task.
//!
//! - Compiler:           IAR EWAVR and GNU GCC for AVR
//! - Supported devices:  AT90USB1287, AT90USB1286, AT90USB647, AT90USB646
//!
//! \author               Atmel Corporation: http://www.atmel.com \n
//!                       Support and FAQ: http://support.atmel.no/
//!
//! ***************************************************************************

/* Copyright (c) 2007, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//_____  I N C L U D E S ___________________________________________________

#include "config.h"
#include "conf_usb.h"
#include "audio_task.h"
#include "lib_board/stk_525/stk_525.h"
#include "lib_mcu/usb/usb_drv.h"
#include "lib_mcu/adc/adc_drv.h"
#include "lib_mcu/timer/timer8_drv.h"
#include "usb_descriptors.h"
#include "modules/usb/device_chap9/usb_standard_request.h"
#include "sound.h"

//_____ M A C R O S ________________________________________________________


//_____ D E F I N I T I O N S ______________________________________________


//_____ D E C L A R A T I O N S ____________________________________________

//! Mute control of micro
//! TRUE when ACTIVE, FALSE otherwise
//!
bit b_micro_mute;

//! Represents the current volume of audio stream micro
//! No effet in this demo
//!
S16 s16_micro_volume;

//! Play position in sample sound storage in flash raw
//!
U16 g_u16_micro_sample_pos=0;


//! This function initializes the USB the associated variables.
//!
void audio_task_init(void)
{
   Leds_init();
   Joy_init();
   Hwb_button_init();
   init_adc();
   
   // Sample rate settings
   // At 8 kHz, signal is be sampled every 125s. When endpoint is full (8 sample words), it is validated.
   // But periodically a real-time problem may occur, since the Host token is also sent approx. every 1ms : it can happen
   // that the token arrives while the Device endpoint has not been validated yet (the difference between these
   // two moments can be some secs), and the packet is not sent : 1 frame (1ms) is missed. So the sample rate has been
   // fixed every 120s, to have a 40s endpoint "free" delay to decrease error cases.
   Timer8_set_waveform_mode(TIMER8_WGM_CTC_OCR);   // Timer 2 mode CTC
   Timer8_set_compare(120);                        // (8MHz/8) * 120s = 120ctn
   Timer8_set_clock(TIMER8_0_CLKIO_BY_8);          // Prescaler div8
   Timer8_compare_it_enable();                     // Enable compare interrupt
   
   b_micro_mute = FALSE;  // default state : enabled
}


//! @brief Entry point of the audio management task
//!
void audio_task(void)
{
   Leds_set_val(MSB(s16_micro_volume));
}


//! @brief Timer2 comparator interrupt routine service
//!
//! Get ADC sample (microphone or alternate external source) and load it to
//! the Isochronous endpoint.
//!
//! @note The dual bank endpoint usage allows to get ride of an extra synchonisation buffer
//!
#ifdef __GNUC__
 ISR(TIMER2_COMPA_vect)
#else
#pragma vector = TIMER2_COMPA_vect
__interrupt void timer2_comp_interrupt()
#endif
{
   S16 sample;
   U8 sav_ep;
      
   if(!Is_device_enumerated())
      return;                 // Device not ready

   sav_ep=Usb_get_selected_endpoint();       // save actually pipe
   Usb_select_endpoint(EP_AUDIO_IN);
   if(!Is_usb_write_enabled())
   {
      Usb_select_endpoint(sav_ep);           // restore pipe
      return;                                // Endpoint not free
   }
   
   if (b_micro_mute == FALSE)
   {
      //** Acquires new sample (from on-board micro, from external source, or from sample in flash raw )
      if( Is_btn_middle() || Is_joy_up() )
      {
         // Play flash raw sample
#ifdef __GNUC__
         LSB(sample) = pgm_read_byte_near(&sample_sound[g_u16_micro_sample_pos++]);
         MSB(sample) = pgm_read_byte_near(&sample_sound[g_u16_micro_sample_pos++]);
#else
         LSB(sample) = sample_sound[g_u16_micro_sample_pos++];
         MSB(sample) = sample_sound[g_u16_micro_sample_pos++];
#endif
         if (g_u16_micro_sample_pos > SAMPLE_SOUND_LEN)
            g_u16_micro_sample_pos=0;  // end of sample sound, then run in a loop
      }
      else if (Is_joy_down())
      {
         // Get external input
         sample = (S32) (64*Get_adc_ext_val()-0x8000);  // EXTERNAL : joystick pushed DOWN
      }
      else
      {
         // All joystick directions released
         // Get micro input
         sample = (S32) (64*Get_adc_mic_val()-0x8000);
      }
      // Send new sample
      Usb_write_byte(LSB(sample));
      Usb_write_byte(MSB(sample));
   }
   else
   {
      //** Mute enable then send new sample null
      Usb_write_byte(0);
      Usb_write_byte(0);
   }

   // If pipe full then send it   
   if (Usb_byte_counter()==EP_SIZE_1)
   {
      Usb_send_in();
   }
   Usb_select_endpoint(sav_ep);              // restore pipe
}


//! @brief  Analog Channel 3 Acquisition routine (ADC)
//!
//! @note   User can choose to transmit a signal that is present on the free PF3 input (available on J7.4 of STK525)
//!         This can be used to do some measures from a pure sine wave sent on PF3 from an external generator.
//!
//! @param none
//!
//! @return none
U16 Get_adc_ext_val(void)
{
   Start_conv_channel(3);
   while (!Is_adc_conv_finished());
   return (U16)(ADCL+((U16)(ADCH<<8)));
}
