/*This file has been prepared for Doxygen automatic documentation generation.*/
//! \file *********************************************************************
//!
//! \brief This file manages the USB Host Audio control application.
//!
//! - Compiler:           IAR EWAVR and GNU GCC for AVR
//! - Supported devices:  AT90USB1287, AT90USB1286, AT90USB647, AT90USB646
//!
//! \author               Atmel Corporation: http://www.atmel.com \n
//!                       Support and FAQ: http://support.atmel.no/
//!
//! ***************************************************************************

/* Copyright (c) 2007, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//_____  I N C L U D E S ___________________________________________________

#include "config.h"
#include "conf_usb.h"
#include "host_audio_task.h"
#include "modules/usb/host_chap9/usb_host_task.h"
#include "modules/usb/host_chap9/usb_host_enum.h"
#include "lib_mcu/usb/usb_drv.h"

//_____ M A C R O S ________________________________________________________

#ifndef LOG_STR_CODE
#define LOG_STR_CODE(str)
#else
U8 code log_audio_connect[]="Audio Connected";
#endif

//_____ D E F I N I T I O N S ______________________________________________

#define   AUDIO_STREAM_SIZE       16


//_____ D E C L A R A T I O N S ____________________________________________

volatile U8  audio_connected;
volatile U8  audio_cpt_sof;
U8  mute;

U8  pipe_audio_in;
U8  interf_audio_stream;

volatile U8  audio_stream_in[AUDIO_STREAM_SIZE];
volatile U8  audio_stream_ptr_wr;
volatile U8  audio_stream_ptr_rd;
volatile U8  audio_stream_size;
volatile bit audio_stream_empty;


//!@brief This function initializes the USB Host Audio application
//!
//!
//!@param none
//!
//!@return none
void host_audio_task_init(void)
{
   Leds_init();
   audio_connected = 0;
   Joy_init();
   
   // D/A converter output rate management
   // At 8 kHz, output should be updated every 125s. But real-time issues may occur between both USB devices :
   // The Host send a token to the Device every millisecond, to get back a new samples frame.
   // On the Device side, this frame is loaded into endpoint every millisecond. So some frames may be missed in
   // the case where the endpoint is not yet validated while the token has been sent.
   // To decrease error cases and probability, the device sample rate has been fixed to 120s.
   // The Host ouput rate is also set to 120s.
   TCCR2A=0x02;         // Prescaler div8
   OCR2A=120;           // Reload value to count 8x120=960 for FOSC 8MHz =>120S refresh rate
   TCCR2B=0x02;         // Timer 2 mode CTC
   TIMSK2|=(1<<OCIE2A); // Enable compare interrupt
   
   // D/A converter hardware configuration
   // In this case, user is free to do what he wants
   Dac_output_init();
}



//! @brief This function manages the USB Host Audio application
//!
//! @param none
//!
//! @return none
void host_audio_task(void)
{
U8 i, j;

   // Audio Device Management
   // *********************
   if(Is_host_ready())
   {
      if(Is_new_device_connection_event())   // Device connection
      {
         for(i=0;i<Get_nb_supported_interface();i++)
         {
            // Audio Streaming Interface
            // Select the audio streaming interface that has an IN PIPE
            if((Get_class(i)==AUDIO_AS_INTERFACE_CLASS) && (Get_subclass(i)==AUDIO_AS_INTERFACE_SUB_CLASS) && (Get_nb_ep(i) != 0))
            {
               for (j=0 ; j<Get_nb_ep(i) ; j++)
               {
                 if (Is_ep_addr_in(Get_ep_addr(i,j)))
                 {
                   // Log in device
                   audio_connected=1;
                   audio_stream_ptr_wr = 0;
                   audio_stream_ptr_rd = 0;
                   audio_stream_empty = TRUE;
                   Led0_on();
                   Host_enable_sof_interrupt();
                   LOG_STR_CODE(log_audio_connect);
                   
                   // Select and enable ISOCHRONOUS pipe
                   pipe_audio_in = host_get_hwd_pipe_nb(Get_ep_addr(i,j));
                   Host_select_pipe(PIPE_AUDIO_IN);
                   Host_continuous_in_mode();
                   Host_unfreeze_pipe();
                   
                   // Set the Device in Mute mode
                   mute = TRUE;
                   Led3_on();   // LED3 is ON when device muted
                   host_audio_set_cur(mute);        // set device default state : muted
                   
                   // Enable alternate streaming interface
                   interf_audio_stream = i;                     // store interface number
                   host_set_interface(interf_audio_stream,1);   // enable streaming interface with "alternate 1" on Device
                   break;
                 }
               }
            }
         }
      }

      if (audio_connected)
      {
         // Check in AUDIO_PIPE_IN for incoming stream
         // ******************************************
         Host_select_pipe(PIPE_AUDIO_IN);
         
         // Error management : clear any error (time-out, CRC, etc...) and unfreeze pipe if error flag rises
         // Error flag rises after three errors. Errors can be identified in UPERRX register.
         if (Is_host_pipe_error())
         {
           Host_ack_all_errors();
           Host_unfreeze_pipe();
         }

         // Stream management
         if (Is_host_in_received() && (Is_host_stall()==FALSE))
         {
           // Pipe has received a new frame !
           Disable_interrupt();   // to avoid interrupt access to audio_stream_in[] buffer
           while ((Host_data_length_U8() != 0) && (audio_stream_ptr_wr != AUDIO_STREAM_SIZE))
           {
              audio_stream_in[audio_stream_ptr_wr] = Host_read_byte();
              audio_stream_ptr_wr++;
           }
           audio_stream_size = audio_stream_ptr_wr;
           audio_stream_empty = FALSE;
           Host_ack_in_received();  // all the pipe data has been read
           Host_send_in();
           Enable_interrupt();
         }
         
         // Mute control : toggle MUTE state with Joystick DOWN direction
         if (Is_joy_down())
         {
           audio_cpt_sof = 0;
           while (audio_cpt_sof != 10); // debounce
           if (mute == TRUE)
           {
             mute=FALSE;
             Led3_off();
           }
           else
           {
             mute=TRUE;
             Led3_on();
           }
           host_audio_set_cur(mute);
           while (Is_joy_down());
           audio_cpt_sof = 0;           // debounce
           while (audio_cpt_sof != 10);
         }
      }
   }

   // Device disconnection...
   if(Is_device_disconnection_event())
   {
      Leds_off();
      audio_stream_empty = TRUE;
      audio_stream_ptr_rd = 0;
      audio_stream_ptr_wr = 0;
      audio_stream_size = 0;
      audio_connected = 0;
   }
}



//! @brief sof_action
//!
//! This function increments the cdc_cpt_sof counter each times
//! the USB Start Of Frame interrupt subroutine is executed (1ms)
//! Usefull to manage time delays
//!
//! @warning Code:?? bytes (function code length)
//!
//! @param none
//!
//! @return none
void sof_action(void)
{
   audio_cpt_sof++;
}




//! @brief Timer2 comparator interrupt routine service
//!
//! When in microphone mode, get adc microphone sample and load it to
//! the Isochronous endpoint.
//!
//! @note The dual bank endpoint usage allows to get ride of an extra synchonisation buffer
//!
//! @param none
//!
//! @return none
#ifdef __GNUC__
 ISR(TIMER2_COMPA_vect)
#else
#pragma vector = TIMER2_COMPA_vect
__interrupt void timer2_comp_interrupt()
#endif
{
   U8   data_dac;
   U16  data_16b;

   if (audio_connected)
   {
      if (audio_stream_empty == FALSE)
      {
        // Get the sample word (16 bits)
        data_16b = audio_stream_in[audio_stream_ptr_rd];
        audio_stream_ptr_rd++;
        data_16b |= ((U16) (audio_stream_in[audio_stream_ptr_rd])) << 8;
        audio_stream_ptr_rd++;
        
        // Decode PCM 16 bits word to 8 bits unsigned
        data_16b ^= 0x8000; // complement MSB
        data_dac = (U8)(data_16b>>8);
        
        // Drives R-2R ladder (uses a complete PORT ; PORTA is free on STK525)
        DAC_SendPort(data_dac);
        
        // Check if end of buffer reached
        if (audio_stream_ptr_rd == audio_stream_size) // no more data to read
        {
          audio_stream_ptr_rd = 0;
          audio_stream_ptr_wr = 0;
          audio_stream_size = 0;
          audio_stream_empty = TRUE;
        }
      }
   }
}



//! @brief  Function that ouputs the 8-bits value to the R-2R ladder (2 configurations possibles)
//!         Refer to the "fig" folder that contains the schematics of the two R-2R ladder configurations
//!
//! @note  This function can also be modified to transmit the 8-bit sound sample to an external DAC (external peripheral, PWM, etc..)
//!
//! @param none
//!
//! @return none
void DAC_SendPort(U8 data_dac)
{
  U8   port_value;
  
  // DAC_OUT value must be set to the value specified by your external DAC ladder configuration
#if   (DAC_OUT==PORT_A_RIGHT)
  // In this configuration, see the R-2R ladder configuration on the "R2R_LADDER_RIGHT" schematic
  PORTA = data_dac;
#elif (DAC_OUT==PORT_A_CPLX)
  // In this configuration (easier to mount directly on EXPAND0 connector...), the connexions of the PORTA signals are not
  // the same that for the R2R_LADDER_RIGHT schematic. The resistor structure is the same, but the port connexions are differents :
  // see R2R_LADDER_CPLX schematic.
  port_value  = (data_dac&0x01);          // bit 0 : portA,0
  port_value |= (data_dac&0x02)<<(2-1);   // bit 1 : portA,2
  port_value |= (data_dac&0x04)<<(4-2);   // bit 2 : portA,4
  port_value |= (data_dac&0x08)<<(6-3);   // bit 3 : portA,6
  port_value |= (data_dac&0x10)<<(7-4);   // bit 4 : portA,7
  port_value |= (data_dac&0x20);          // bit 5 : portA,5
  port_value |= (data_dac&0x40)>>(6-3);   // bit 6 : portA,3
  port_value |= (data_dac&0x80)>>(7-1);   // bit 7 : portA,1
  PORTA = port_value;
#endif
}


