/**
 * \file
 *
 * Copyright (C) 2009 Atmel Corporation. All rights reserved.
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/*!
 * \file
 *
 * \brief USB Device Controller interface
 *
 * This file contains functions and data structures common to all USB
 * Device Controller implementations.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a USBB module can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Copyright (C) 2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef USB_UDC_H_INCLUDED
#define USB_UDC_H_INCLUDED

#include <bitops.h>
#include <usb/usb_protocol.h>
#include <app/config_usb.h>

struct usb_request;

/**
 * Endpoint ID. Used to identify endpoints on a USB device.
 */
typedef int	usb_ep_id_t;

/**
 * \brief Bit definitions for udc::flags
 */
enum usb_device_flag {
	USB_DEV_IS_ENABLED,	//!< Device Controller enabled
	USB_DEV_AUTOATTACH,	//!< Attach when possible
	USB_DEV_HAS_POWER,	//!< Vbus detected
	USB_DEV_IS_SUSPENDED,	//!< Bus is suspended
};

/**
 * Data about a USB Device Controller usable by the rest of the
 * system.
 */
struct udc {
	/**
	 * Current device configuration descriptor or NULL if the
	 * device has not reached the CONFIGURED state.
	 */
	const struct usb_configuration_descriptor *config;
	/** The speed that we're currently operating at. */
	enum usb_device_speed	speed;
	/** Bitmask of flags indicating the state of the device */
	unsigned long		flags;
	/**
	 * Our current device address. 0 indicates that we haven't
	 * yet reached the ADDRESS state.
	 */
	uint8_t			address;
};

/**
 * \brief Test whether the USB Device Controller is running at high
 * speed.
 *
 * If the UDC doesn't support high-speed operation, this function will
 * always return false. This may be used by the compiler to eliminate
 * high-speed only code when the UDC doesn't support it.
 *
 * \retval true The UDC is running at high speed
 * \retval false The UDC is not running at high speed
 */
static inline bool udc_is_high_speed(struct udc *udc)
{
#ifdef CONFIG_UDC_HIGH_SPEED
	return udc->speed == USB_SPEED_HIGH;
#else
	return false;
#endif
}

/**
 * \brief Test whether the USB Device Controller is running at full
 * speed.
 *
 * If the UDC doesn't support full-speed operation, this function will
 * always return false. This may be used by the compiler to eliminate
 * full-speed only code when the UDC doesn't support it, e.g. when it is
 * configured to run in low-speed mode only.
 *
 * \retval true The UDC is running at full speed
 * \retval false The UDC is not running at full speed
 */
static inline bool udc_is_full_speed(struct udc *udc)
{
#ifdef CONFIG_UDC_FULL_SPEED
	return udc->speed == USB_SPEED_FULL;
#else
	return false;
#endif
}

/**
 * \brief Test whether the USB Device Controller is running at low
 * speed.
 *
 * If the UDC doesn't support low-speed operation, this function will
 * always return false. This may be used by the compiler to eliminate
 * low-speed only code when the UDC doesn't support it. Note that
 * low-speed mode is incompatible with all other modes, so this function
 * will always return false for full- and/or high-speed capable devices.
 *
 * \retval true The UDC is running at low speed
 * \retval false The UDC is not running at low speed
 */
static inline bool udc_is_low_speed(struct udc *udc)
{
#ifdef CONFIG_UDC_LOW_SPEED
	return udc->speed == USB_SPEED_LOW;
#else
	return false;
#endif
}

/**
 * Figure out which usb_device_state we're in. This function is
 * probably overly expensive for most purposes, but might useful for
 * debugging. It's also useful for documentation purposes.
 */
static inline enum usb_device_state udc_state(struct udc *udc)
{
	int state;

	if (!test_bit(USB_DEV_HAS_POWER, &udc->flags))
		state = USB_STATE_ATTACHED;
	else if (udc->speed == USB_SPEED_UNKNOWN)
		state = USB_STATE_POWERED;
	else if (!udc->address)
		state = USB_STATE_DEFAULT;
	else if (!udc->config)
		state = USB_STATE_ADDRESS;
	else
		state = USB_STATE_CONFIGURED;

	if (test_bit(USB_DEV_IS_SUSPENDED, &udc->flags))
		state |= USB_STATE_SUSPENDED;

	return state;
}

extern struct udc *udc_init(void);

extern void udc_attach(struct udc *udc);
extern void udc_detach(struct udc *udc);

extern usb_ep_id_t udc_ep_create(struct udc *udc,
		const struct usb_endpoint_descriptor *desc,
		unsigned int nr_banks);
extern void udc_ep_destroy(struct udc *udc, usb_ep_id_t ep);

extern int udc_ep_clear_halt(struct udc *udc, usb_ep_id_t ep);
extern int udc_ep_set_halt(struct udc *udc, usb_ep_id_t ep);
extern int udc_ep_is_halted(struct udc *udc, usb_ep_id_t ep);

extern void udc_ep_clear_wedge(struct udc *udc, usb_ep_id_t ep);
extern void udc_ep_set_wedge(struct udc *udc, usb_ep_id_t ep);
extern bool udc_ep_is_wedged(struct udc *udc, usb_ep_id_t);

extern void udc_ep0_submit_out_req(struct udc *udc, struct usb_request *req);
extern void udc_ep0_submit_in_req(struct udc *udc, struct usb_request *req);

extern int udc_ep0_write_sync(struct udc *udc, const void *data, size_t len);
extern void udc_ep0_send_status(struct udc *udc);
extern void udc_ep0_expect_status(struct udc *udc);
extern void udc_set_address(struct udc *udc, unsigned int addr);

extern void udc_ep_submit_out_req(struct udc *udc, usb_ep_id_t ep,
		struct usb_request *req);
extern void udc_ep_submit_in_req(struct udc *udc, usb_ep_id_t ep,
		struct usb_request *req);

extern void udc_ep_flush(struct udc *udc, usb_ep_id_t ep);

extern int udc_enter_test_mode(struct udc *udc, unsigned int mode);

extern int udc_lib_process_setup_request(struct udc *udc,
		struct usb_setup_req *req);

#endif /* USB_UDC_H_INCLUDED */
