/**
 * \file
 *
 * Copyright (C) 2009 Atmel Corporation. All rights reserved.
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/*!
 * \file
 *
 * \brief USB device function core layer interface.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices with a USB Device Controller
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Copyright (C) 2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef USB_FUNC_CORE_H_INCLUDED
#define USB_FUNC_CORE_H_INCLUDED

#include <buffer.h>

struct usb_func_iface;
struct usb_configuration_descriptor;
struct usb_setup_req;
struct udc;

/**
 * \brief USB device interface driver operations.
 */
struct usb_func_iface_ops {
	/**
	 * \brief Enable an interface.
	 *
	 * \param udc The USB Device Controller.
	 * \param iface The interface to be enabled.
	 *
	 * \retval 0 The interface was successfully enabled.
	 * \retval <0 The interface could not be enabled.
	 */
	int	(*enable)(struct udc *udc, struct usb_func_iface *iface);
	/**
	 * \brief Disable an interface.
	 *
	 * \param udc The USB Device Controller.
	 * \param iface The interface to be disabled.
	 */
	void	(*disable)(struct udc *udc, struct usb_func_iface *iface);
	/**
	 * \brief Handle a control request directed at an interface.
	 *
	 * \param udc The USB Device Controller.
	 * \param iface The interface which the request is for.
	 * \param req The SETUP packet.
	 *
	 * \retval 0 The control request was successfully handled.
	 * \retval <0 The control request was not handled, and the
	 *	default control endpoint should be halted.
	 */
	int	(*setup)(struct udc *udc, struct usb_func_iface *iface,
			struct usb_setup_req *req);
};

/**
 * \brief An alternate setting of an interface.
 */
struct usb_func_iface_setting {
	/** \brief Interface operations for this setting */
	const struct usb_func_iface_ops	*ops;

	/** \brief Interface descriptor when running at full speed */
	const struct usb_interface_descriptor	*fs_desc;
	/** \brief Size of fs_desc, including sub-descriptors */
	size_t				fs_desc_size;
	/** \brief Interface descriptor when running at high speed */
	const struct usb_interface_descriptor	*hs_desc;
	/** \brief Size of hs_desc, including sub-descriptors */
	size_t				hs_desc_size;
	/** \brief Buffer for use when sending the descriptor */
	struct buffer			desc_buf;
};

/**
 * \brief A USB device interface.
 */
struct usb_func_iface {
	/** \brief The current setting of this interface. */
	struct usb_func_iface_setting	*cur_setting;
	/** \brief The number of alternate settings for this interface. */
	size_t				nr_settings;
	/** \brief The available alternate settings for this interface. */
	struct usb_func_iface_setting	setting[1];
};

/**
 * \brief A USB device configuration.
 *
 * A configuration is a set of interfaces that can be selected using
 * the SET_CONFIGURATION standard control request.
 */
struct usb_func_config {
	/**
	 * \brief The interfaces available when this configuration is
	 * active.
	 */
	struct usb_func_iface		**interface;
	/** \brief The configuration descriptor for this configuration. */
	struct usb_configuration_descriptor *desc;
};

extern void usb_func_add_interface(struct usb_func_config *config,
		struct usb_func_iface *iface);
extern struct usb_func_config *usb_func_add_config(
		struct usb_configuration_descriptor *desc);

#endif /* USB_FUNC_CORE_H_INCLUDED */
