/**
 * \file
 *
 * \brief Standard string operations.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#ifndef STRING_H_INCLUDED
#define STRING_H_INCLUDED

#include <types.h>

unsigned long strtoul(const char *s, char **endptr, int base);
long strtol(const char *s, char **endptr, int base);

/**
 * \brief Copy memory area.
 *
 * Copy \a n bytes from \a src to \a dest.
 *
 * \param dest Pointer to the destination memory area.
 * \param src Pointer to the source memory area.
 * \param n The number of bytes to copy.
 * \return The \a dest parameter.
 */
void *memcpy(void *dest, const void *src, size_t n);

/**
 * \brief Initialize memory area.
 *
 * Fill the first \a n bytes of the memory area \a s with the constant
 * byte value \a c.
 *
 * \param s Pointer to the memory area to be filled.
 * \param c The byte value to use as fill value.
 * \param n The number of bytes to initialize.
 * \return The \a s parameter.
 */
void *memset(void *s, int c, size_t n);

/**
 * \brief Search a memory area for a character.
 *
 * Scan the memory area from \a s to \a s + \a n (non-inclusive) for
 * the occurrence of \a c.
 *
 * \param s Pointer to start of memory area.
 * \param c Character to look for.
 * \param n Size of memory area.
 * \return Pointer to the matching character or NULL if \a c doesn't
 *	occur in the given memory area.
 */
void *memchr(const void *s, int c, size_t n);

/**
 * \brief Compare the two strings \a s1 and \a s2.
 *
 * \param s1 A NUL-terminated string.
 * \param s2 Another NUL-terminated string.
 * \retval <0 \a s1 is less than \a s2.
 * \retval 0 \a s1 equals \a s2.
 * \retval >0 \a s1 is greater than \a s2.
 */
int strcmp(const char *s1, const char *s2);

/**
 * \brief Calculate the length of the string \a s.
 *
 * \param s A NUL-terminated string.
 * \return The number of characters in \a s before the terminating
 *	'\\0' character.
 */
size_t strlen(const char *s);

/**
 * \brief Locate the character \a c in the string \a s.
 *
 * \param s A NUL-terminated string.
 * \param c The character to look for.
 *
 * \return A pointer to the first occurrence of the character \a c in
 *	the string \a s or NULL if \a c was not found.
 */
char *strchr(const char *s, int c);

#endif /* STRING_H_INCLUDED */
