/**
 * \file
 *
 * \brief Power Manager v3 register definitions
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  Any device with a PM v3 module
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#ifndef HARDWARE_PM_V3_REGS_H_INCLUDED
#define HARDWARE_PM_V3_REGS_H_INCLUDED

/* PM v3 registers and bitfield definitions */
#define PM_MCCTRL			0x0000	/* Main Clock Control */
# define PM_MCCTRL_MCSEL_MASK		(3   <<  0)	/* Main clock select */
# define PM_MCCTRL_MCSEL(x)		((x) <<  0)
# define PM_MCCTRL_MCSEL_SLOW		(0   <<  0)	/* Slow clock */
# define PM_MCCTRL_MCSEL_OSC0		(1   <<  0)	/* OSC0 */
# define PM_MCCTRL_MCSEL_PLL0		(2   <<  0)	/* PLL0 */
# define PM_MCCTRL_OSCEN(x)		(1 << ((x) + 2)) /* OSCx enable */
# define PM_MCCTRL_OSCEN_BIT(x)		((x) + 2)	/* OSCx enable */
# define PM_MCCTRL_CRIPEL_BIT		24		/* CRIPEL enable */
#define PM_CKSEL			0x0004	/* Clock Select */
# define PM_CKSEL_CPUSEL(x)		((x) <<  0)	/* CPU clock divider */
# define PM_CKSEL_CPUDIV		(1   <<  7)	/* CPU div enable */
# define PM_CKSEL_HSBSEL(x)		((x) <<  8)	/* HSB clock divider */
# define PM_CKSEL_HSBDIV		(1   << 15)	/* HSB div enable */
# define PM_CKSEL_PBASEL(x)		((x) << 16)	/* PBA clock divider */
# define PM_CKSEL_PBADIV		(1   << 23)	/* PBA div enable */
# define PM_CKSEL_PBBSEL(x)		((x) << 24)	/* PBB clock divider */
# define PM_CKSEL_PBBDIV		(1   << 31)	/* PBB div enable */
#define PM_CPUMASK			0x0008	/* CPU clock mask */
#define PM_HSBMASK			0x000c	/* HSB clock mask */
#define PM_PBAMASK			0x0010	/* PBA clock mask */
#define PM_PBBMASK			0x0014	/* PBB clock mask */
#define PM_PBADIVMASK			0x0018	/* PBA div clock mask */
#define PM_PBBDIVMASK			0x001c	/* PBB div clock mask */
#define PM_PLL(x)	(0x0040 + 4 * (x))	/* PLL control */
# define PM_PLL_PLLEN			(1   <<  0)	/* PLL enable */
# define PM_PLL_PLLOSC(x)		((x) <<  1)	/* Osc select */
# define PM_PLL_GET_PLLOSC(reg)		(((reg) >> 1) & 3)
# define PM_PLL_VCO_MHZ_MASK		(3   <<  3)	/* VCO frequency */
# define PM_PLL_VCO_MHZ_25_50		(0   <<  3)	/* 25-50 MHz */
# define PM_PLL_VCO_MHZ_50_100		(1   <<  3)	/* 50-100 MHz */
# define PM_PLL_VCO_MHZ_100_200		(2   <<  3)	/* 100-200 MHz */
# define PM_PLL_VCO_MHZ_200_400		(3   <<  3)	/* 200-400 MHz */
# define PM_PLL_OUTPUT_DIV_BY_2		(1   <<  5)	/* Divide by 2 */
# define PM_PLL_PLLDIV(x)		((x) <<  8)	/* PLL divider */
# define PM_PLL_GET_PLLDIV(reg)		(((reg) >> 8) & 0x3f)
# define PM_PLL_PLLMUL(x)		((x) << 16)	/* PLL multiplier */
# define PM_PLL_GET_PLLMUL(reg)		(((reg) >> 16) & 0x3f)
# define PM_PLL_PLLCOUNT(x)		((x) << 24)	/* PLL startup count */
#define PM_OSCCTRL(x)	(0x0080 + 4 * (x))	/* Oscillator control */
# define PM_OSC_MODE_MASK		(7   <<  0)	/* Oscillator Mode */
# define PM_OSC_MODE_EXT		(0   <<  0)	/* External clock */
# define PM_OSC_MODE_XTAL_1_12		(1   <<  0)	/* 1-12 MHz crystal */
# define PM_OSC_MODE_XTAL_12_24		(2   <<  0)	/* 12-24 MHz crystal */
# define PM_OSC_MODE_XTAL_24_42		(3   <<  0)	/* 24-42 MHz crystal */
# define PM_OSC_MODE_XTAL_42_66		(4   <<  0)	/* 42-66 MHz crystal */
# define PM_OSC_MODE_FEEDBACK		(1   <<  3)	/* Feedback resistor */
# define PM_OSC_STARTUP_MASK		(7   <<  8)	/* Startup cycles */
# define PM_OSC_STARTUP_0		(0   <<  8)	/* 0 cycles */
# define PM_OSC_STARTUP_32		(1   <<  8)	/* 32 cycles */
# define PM_OSC_STARTUP_64		(2   <<  8)	/* 64 cycles */
# define PM_OSC_STARTUP_512		(3   <<  8)	/* 512 cycles */
# define PM_OSC_STARTUP_1024		(4   <<  8)	/* 1024 cycles */
# define PM_OSC_STARTUP_4096		(5   <<  8)	/* 4096 cycles */
# define PM_OSC_STARTUP_8192		(6   <<  8)	/* 8192 cycles */
# define PM_OSC_STARTUP_16384		(7   <<  8)	/* 16384 cycles */
#define PM_IER				0x00c0	/* Interrupt Enable */
#define PM_IDR				0x00c4	/* Interrupt Disable */
#define PM_IMR				0x00c8	/* Interrupt Mask */
#define PM_ISR				0x00cc	/* Interrupt Status */
#define PM_ICR				0x00d0	/* Interrupt Clear */
#define PM_POSCSR			0x00d4	/* Power/Osc Status */
# define PM_OSCRDY(x)			(1  << (x))	/* OSCx ready */
# define PM_OSCRDY_BIT(x)		(x)
# define PM_OSC32RDY			(1   <<  7)	/* OSC32 ready */
# define PM_LOCK(x)			(1 << ((x) + 8)) /* PLLx locked */
# define PM_LOCKLOST(x)			(1 << ((x) + 16)) /* PLLx lock lost */
# define PM_CKRDY			(1   << 24)	/* Clock ready */
# define PM_MSKRDY			(1   << 25)	/* Mask ready */
# define PM_WAKE			(1   << 26)	/* ??? */
# define PM_PERRDY			(1   << 28)	/* Periph ready */
#define PM_GCCTRL(x)	(0x0100 + 4 * (x))	/* Generic Clock Control */
# define PM_GCCTRL_CEN			(1   <<  0)	/* Clock enable */
# define PM_GCCTRL_DIVEN		(1   <<  1)	/* Divider enable */
# define PM_GCCTRL_OSCSEL(x)		((x) <<  8)	/* Clock source */
# define PM_GCCTRL_OSCSEL_MASK		(3   << 15)
# define PM_GCCTRL_DIV(x)		((x) << 16)	/* Divider */
#define PM_PPCR				0x0150	/* Periph Power Control */
#define PM_RC_RCAUSE			0x0180	/* Reset Cause */
#define PM_WCAUSE			0x0184	/* Wake Cause */
#define PM_AWEN				0x0188	/* Async Wake Enable */

/* Register access macros */
#define pm_read_reg(reg)					\
	mmio_read32((void *)PM_BASE + PM_##reg)
#define pm_write_reg(reg, value)				\
	mmio_write32((void *)PM_BASE + PM_##reg, value)

#endif /* HARDWARE_PM_V3_REGS_H_INCLUDED */
