/**
 * \file
 *
 * \brief Power Manager v2 register definitions
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  Any device with a PM v2 module
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#ifndef HARDWARE_PM_V2_REGS_H_INCLUDED
#define HARDWARE_PM_V2_REGS_H_INCLUDED

/* Register offsets */
#define PM_MCCTRL			0x0000	/* Main Clock Control */
# define PM_MCCTRL_MCSEL_MASK		(3   <<  0)	/* Main clock select */
# define PM_MCCTRL_MCSEL(x)		((x) <<  0)
# define PM_MCCTRL_MCSEL_SLOW		(0   <<  0)	/* Slow clock */
# define PM_MCCTRL_MCSEL_OSC0		(1   <<  0)	/* OSC0 */
# define PM_MCCTRL_MCSEL_PLL0		(2   <<  0)	/* PLL0 */
# define PM_MCCTRL_OSCEN(x)		(1 << ((x) + 2)) /* OSCx enable */
# define PM_MCCTRL_OSCEN_BIT(x)		((x) + 2)	/* OSCx enable */
#define PM_CKSEL			0x0004	/* Clock Select */
# define PM_CKSEL_CPUSEL(x)		((x) <<  0)	/* CPU clock divider */
# define PM_CKSEL_CPUDIV		(1   <<  7)	/* CPU div enable */
# define PM_CKSEL_HSBSEL(x)		((x) <<  8)	/* HSB clock divider */
# define PM_CKSEL_HSBDIV		(1   << 15)	/* HSB div enable */
# define PM_CKSEL_PBASEL(x)		((x) << 16)	/* PBA clock divider */
# define PM_CKSEL_PBADIV		(1   << 23)	/* PBA div enable */
# define PM_CKSEL_PBBSEL(x)		((x) << 24)	/* PBB clock divider */
# define PM_CKSEL_PBBDIV		(1   << 31)	/* PBB div enable */
#define PM_CPUMASK			0x0008	/* CPU clock mask */
#define PM_HSBMASK			0x000c	/* HSB clock mask */
#define PM_PBAMASK			0x0010	/* PBA clock mask */
#define PM_PBBMASK			0x0014	/* PBB clock mask */
#define PM_PLL(x)	(0x0020 + 4 * (x))	/* PLL control */
# define PM_PLL_PLLEN			(1   <<  0)	/* PLL enable */
# define PM_PLL_PLLOSC(x)		((x) <<  1)	/* Osc select */
# define PM_PLL_GET_PLLOSC(reg)		(((reg) >> 1) & 1)
# define PM_PLL_VCO_MHZ_MASK		(1   <<  2)	/* VCO frequency */
# define PM_PLL_VCO_MHZ_160_240		(0   <<  2)	/* 160-240 MHz */
# define PM_PLL_VCO_MHZ_80_180		(1   <<  2)	/* 80-180 MHz */
# define PM_PLL_OUTPUT_DIV_BY_2		(1   <<  3)	/* Divide by 2 */
# define PM_PLL_WIDE_BANDWIDTH_DIS	(1   <<  4)	/* Wide bandwidth */
# define PM_PLL_PLLDIV(x)		((x) <<  8)	/* PLL divider */
# define PM_PLL_GET_PLLDIV(reg)		(((reg) >> 8) & 0xff)
# define PM_PLL_PLLMUL(x)		((x) << 16)	/* PLL multiplier */
# define PM_PLL_GET_PLLMUL(reg)		(((reg) >> 16) & 0xff)
# define PM_PLL_PLLCOUNT(x)		((x) << 24)	/* PLL startup count */
#define PM_OSCCTRL(x)	(0x0028 + 4 * (x))
# define PM_OSC_MODE_MASK		(7   <<  0)	/* Oscillator Mode */
# define PM_OSC_MODE_EXT		(0   <<  0)	/* External clock */
# define PM_OSC_MODE_XTAL		(1   <<  0)	/* Crystal */
# define PM_OSC_STARTUP_MASK		(7   <<  8)	/* Startup cycles */
# define PM_OSC_STARTUP_0		(0   <<  8)	/* 0 cycles */
# define PM_OSC_STARTUP_64		(1   <<  8)	/* 64 cycles */
# define PM_OSC_STARTUP_128		(2   <<  8)	/* 128 cycles */
# define PM_OSC_STARTUP_2048		(3   <<  8)	/* 2048 cycles */
# define PM_OSC_STARTUP_4096		(4   <<  8)	/* 4096 cycles */
# define PM_OSC_STARTUP_8192		(5   <<  8)	/* 8192 cycles */
# define PM_OSC_STARTUP_16384		(6   <<  8)	/* 16384 cycles */
#define PM_OSCCTRL32			0x0030
# define PM_OSC32_EN			(1   <<  0)	/* OSC32 enable */
# define PM_OSC32_MODE_MASK		(7   <<  8)	/* Oscillator Mode */
# define PM_OSC32_MODE_EXT		(0   <<  8)	/* External clock */
# define PM_OSC32_MODE_XTAL		(1   <<  8)	/* Crystal */
# define PM_OSC32_STARTUP_MASK		(7   << 16)	/* Startup cycles */
# define PM_OSC32_STARTUP_0		(0   << 16)	/* 0 cycles */
# define PM_OSC32_STARTUP_128		(1   << 16)	/* 128 cycles */
# define PM_OSC32_STARTUP_8192		(2   << 16)	/* 8192 cycles */
# define PM_OSC32_STARTUP_16384		(3   << 16)	/* 16384 cycles */
# define PM_OSC32_STARTUP_65536		(4   << 16)	/* 65536 cycles */
# define PM_OSC32_STARTUP_131072	(5   << 16)	/* 131072 cycles */
# define PM_OSC32_STARTUP_262144	(6   << 16)	/* 262144 cycles */
# define PM_OSC32_STARTUP_524288	(7   << 16)	/* 524288 cycles */
#define PM_IER				0x0040
#define PM_IDR				0x0044
#define PM_IMR				0x0048
#define PM_ISR				0x004c
#define PM_ICR				0x0050
#define PM_POSCSR			0x0054
# define PM_LOCK(x)			(1 << (x))	/* PLLx locked */
# define PM_CKRDY			(1   <<  5)	/* Clock ready */
# define PM_MSKRDY			(1   <<  6)	/* Mask ready */
# define PM_OSCRDY(x)			(1  << ((x) + 7)) /* OSCx ready */
# define PM_OSCRDY_BIT(x)		((x) + 7)
# define PM_OSC32RDY			(1   <<  9)	/* OSC32 ready */
# define PM_BODDET			(1   << 16)	/* Brown-out detected */
#define PM_GCCTRL(x)	(0x0060 + 4 * (x))	/* Generic Clock Control */
# define PM_GCCTRL_OSCSEL(x)		((x) <<  0)	/* Clock source */
# define PM_GCCTRL_OSCSEL_MASK		(3   <<  0)
# define PM_GCCTRL_CEN			(1   <<  2)	/* Clock enable */
# define PM_GCCTRL_DIVEN		(1   <<  3)	/* Divider enable */
# define PM_GCCTRL_DIV(x)		((x) <<  8)	/* Divider */
#define PM_RCCR				0x00c0
#define PM_BGCR				0x00c4
#define PM_VREGCR			0x00c8
#define PM_BOD				0x00d0
#define PM_RCAUSE			0x0140		/* Reset Cause */
#define PM_AWEN				0x0144
#define PM_GPLP				0x0200

/* Register access macros */
#define pm_read_reg(reg)					\
	mmio_read32((void *)PM_BASE + PM_##reg)
#define pm_write_reg(reg, value)				\
	mmio_write32((void *)PM_BASE + PM_##reg, value)

#endif /* HARDWARE_PM_V2_REGS_H_INCLUDED */
