/**
 * \file
 *
 * \brief USART register definitions
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices with a USART module
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#ifndef USART_REGS_H_INCLUDED
#define USART_REGS_H_INCLUDED

#include <types.h>

/* USART register offsets */
struct usart_regs {
	uint32_t	CR;
	uint32_t	MR;
	uint32_t	IER;
	uint32_t	IDR;
	uint32_t	IMR;
	uint32_t	CSR;
	uint32_t	RHR;
	uint32_t	THR;
	uint32_t	BRGR;
	uint32_t	RTOR;
	uint32_t	TTGR;
	uint32_t	__reserved0[5];
	uint32_t	FIDI;
	uint32_t	NER;
	uint32_t	XXR;
	uint32_t	IFR;
};

/* Bitfields in CR */
#define USART_RSTRX_OFFSET			2
#define USART_RSTRX_SIZE			1
#define USART_RSTTX_OFFSET			3
#define USART_RSTTX_SIZE			1
#define USART_RXEN_OFFSET			4
#define USART_RXEN_SIZE				1
#define USART_RXDIS_OFFSET			5
#define USART_RXDIS_SIZE			1
#define USART_TXEN_OFFSET			6
#define USART_TXEN_SIZE				1
#define USART_TXDIS_OFFSET			7
#define USART_TXDIS_SIZE			1
#define USART_RSTSTA_OFFSET			8
#define USART_RSTSTA_SIZE			1
#define USART_STTBRK_OFFSET			9
#define USART_STTBRK_SIZE			1
#define USART_STPBRK_OFFSET			10
#define USART_STPBRK_SIZE			1
#define USART_STTTO_OFFSET			11
#define USART_STTTO_SIZE			1
#define USART_SENDA_OFFSET			12
#define USART_SENDA_SIZE			1
#define USART_RSTIT_OFFSET			13
#define USART_RSTIT_SIZE			1
#define USART_RSTNACK_OFFSET			14
#define USART_RSTNACK_SIZE			1
#define USART_RETTO_OFFSET			15
#define USART_RETTO_SIZE			1
#define USART_DTREN_OFFSET			16
#define USART_DTREN_SIZE			1
#define USART_DTRDIS_OFFSET			17
#define USART_DTRDIS_SIZE			1
#define USART_RTSEN_OFFSET			18
#define USART_RTSEN_SIZE			1
#define USART_RTSDIS_OFFSET			19
#define USART_RTSDIS_SIZE			1
#define USART_COMM_TX_OFFSET			30
#define USART_COMM_TX_SIZE			1
#define USART_COMM_RX_OFFSET			31
#define USART_COMM_RX_SIZE			1

/* Bitfields in MR */
#define USART_MODE_OFFSET			0
#define USART_MODE_SIZE				4
#define USART_USCLKS_OFFSET			4
#define USART_USCLKS_SIZE			2
#define USART_CHRL_OFFSET			6
#define USART_CHRL_SIZE				2
#define USART_SYNC_OFFSET			8
#define USART_SYNC_SIZE				1
#define USART_PAR_OFFSET			9
#define USART_PAR_SIZE				3
#define USART_NBSTOP_OFFSET			12
#define USART_NBSTOP_SIZE			2
#define USART_CHMODE_OFFSET			14
#define USART_CHMODE_SIZE			2
#define USART_MSBF_OFFSET			16
#define USART_MSBF_SIZE				1
#define USART_MODE9_OFFSET			17
#define USART_MODE9_SIZE			1
#define USART_CLKO_OFFSET			18
#define USART_CLKO_SIZE				1
#define USART_OVER_OFFSET			19
#define USART_OVER_SIZE				1
#define USART_INACK_OFFSET			20
#define USART_INACK_SIZE			1
#define USART_DSNACK_OFFSET			21
#define USART_DSNACK_SIZE			1
#define USART_MAX_ITERATION_OFFSET		24
#define USART_MAX_ITERATION_SIZE		3
#define USART_FILTER_OFFSET			28
#define USART_FILTER_SIZE			1

/* Bitfields in CSR */
#define USART_RXRDY_OFFSET			0
#define USART_RXRDY_SIZE			1
#define USART_TXRDY_OFFSET			1
#define USART_TXRDY_SIZE			1
#define USART_RXBRK_OFFSET			2
#define USART_RXBRK_SIZE			1
#define USART_ENDRX_OFFSET			3
#define USART_ENDRX_SIZE			1
#define USART_ENDTX_OFFSET			4
#define USART_ENDTX_SIZE			1
#define USART_OVRE_OFFSET			5
#define USART_OVRE_SIZE				1
#define USART_FRAME_OFFSET			6
#define USART_FRAME_SIZE			1
#define USART_PARE_OFFSET			7
#define USART_PARE_SIZE				1
#define USART_TIMEOUT_OFFSET			8
#define USART_TIMEOUT_SIZE			1
#define USART_TXEMPTY_OFFSET			9
#define USART_TXEMPTY_SIZE			1
#define USART_ITERATION_OFFSET			10
#define USART_ITERATION_SIZE			1
#define USART_TXBUFE_OFFSET			11
#define USART_TXBUFE_SIZE			1
#define USART_RXBUFF_OFFSET			12
#define USART_RXBUFF_SIZE			1
#define USART_NACK_OFFSET			13
#define USART_NACK_SIZE				1
#define USART_RIIC_OFFSET			16
#define USART_RIIC_SIZE				1
#define USART_DSRIC_OFFSET			17
#define USART_DSRIC_SIZE			1
#define USART_DCDIC_OFFSET			18
#define USART_DCDIC_SIZE			1
#define USART_CTSIC_OFFSET			19
#define USART_CTSIC_SIZE			1
#define USART_RI_OFFSET				20
#define USART_RI_SIZE				1
#define USART_DSR_OFFSET			21
#define USART_DSR_SIZE				1
#define USART_DCD_OFFSET			22
#define USART_DCD_SIZE				1
#define USART_CTS_OFFSET			23
#define USART_CTS_SIZE				1

/* Bitfields in RHR */
#define USART_RXCHR_OFFSET			0
#define USART_RXCHR_SIZE			9

/* Bitfields in THR */
#define USART_TXCHR_OFFSET			0
#define USART_TXCHR_SIZE			9

/* Bitfields in BRGR */
#define USART_CD_OFFSET				0
#define USART_CD_SIZE				16
#define USART_FP_OFFSET				16
#define USART_FP_SIZE				3

/* Bitfields in RTOR */
#define USART_TO_OFFSET				0
#define USART_TO_SIZE				16

/* Bitfields in TTGR */
#define USART_TG_OFFSET				0
#define USART_TG_SIZE				8

/* Bitfields in FIDI */
#define USART_FI_DI_RATIO_OFFSET		0
#define USART_FI_DI_RATIO_SIZE			11

/* Bitfields in NER */
#define USART_NB_ERRORS_OFFSET			0
#define USART_NB_ERRORS_SIZE			8

/* Bitfields in XXR */
#define USART_XOFF_OFFSET			0
#define USART_XOFF_SIZE				8
#define USART_XON_OFFSET			8
#define USART_XON_SIZE				8

/* Bitfields in IFR */
#define USART_IRDA_FILTER_OFFSET		0
#define USART_IRDA_FILTER_SIZE			8

/* Bitfields in RCR */
#define USART_RXCTR_OFFSET			0
#define USART_RXCTR_SIZE			16

/* Bitfields in TCR */
#define USART_TXCTR_OFFSET			0
#define USART_TXCTR_SIZE			16

/* Bitfields in RNCR */
#define USART_RXNCR_OFFSET			0
#define USART_RXNCR_SIZE			16

/* Bitfields in TNCR */
#define USART_TXNCR_OFFSET			0
#define USART_TXNCR_SIZE			16

/* Bitfields in PTCR */
#define USART_RXTEN_OFFSET			0
#define USART_RXTEN_SIZE			1
#define USART_RXTDIS_OFFSET			1
#define USART_RXTDIS_SIZE			1
#define USART_TXTEN_OFFSET			8
#define USART_TXTEN_SIZE			1
#define USART_TXTDIS_OFFSET			9
#define USART_TXTDIS_SIZE			1

/* Constants for USART_MODE */
#define USART_MODE_NORMAL			0
#define USART_MODE_RS485			1
#define USART_MODE_HARDWARE			2
#define USART_MODE_MODEM			3
#define USART_MODE_ISO7816_T0			4
#define USART_MODE_ISO7816_T1			6
#define USART_MODE_IRDA				8

/* Constants for USCLKS */
#define USART_USCLKS_MCK			0
#define USART_USCLKS_MCK_DIV			1
#define USART_USCLKS_SCK			3

/* Constants for CHRL */
#define USART_CHRL_5				0
#define USART_CHRL_6				1
#define USART_CHRL_7				2
#define USART_CHRL_8				3

/* Constants for PAR */
#define USART_PAR_EVEN				0
#define USART_PAR_ODD				1
#define USART_PAR_SPACE				2
#define USART_PAR_MARK				3
#define USART_PAR_NONE				4
#define USART_PAR_MULTI				6

/* Constants for NBSTOP */
#define USART_NBSTOP_1				0
#define USART_NBSTOP_1_5			1
#define USART_NBSTOP_2				2

/* Constants for CHMODE */
#define USART_CHMODE_NORMAL			0
#define USART_CHMODE_ECHO			1
#define USART_CHMODE_LOCAL_LOOP			2
#define USART_CHMODE_REMOTE_LOOP		3

/* Constants for MSBF */
#define USART_MSBF_LSBF				0
#define USART_MSBF_MSBF				1

/* Constants for OVER */
#define USART_OVER_X16				0
#define USART_OVER_X8				1

/* Constants for CD */
#define USART_CD_DISABLE			0
#define USART_CD_BYPASS				1

/* Constants for TO */
#define USART_TO_DISABLE			0

/* Constants for TG */
#define USART_TG_DISABLE			0

/* Constants for FI_DI_RATIO */
#define USART_FI_DI_RATIO_DISABLE		0

/* Bit manipulation macros */
#define USART_BIT(name)					\
	(1 << USART_##name##_OFFSET)
#define USART_BF(name,value)				\
	(((value) & ((1 << USART_##name##_SIZE) - 1))	\
	 << USART_##name##_OFFSET)
#define USART_BFEXT(name,value)				\
	(((value) >> USART_##name##_OFFSET)		\
	 & ((1 << USART_##name##_SIZE) - 1))
#define USART_BFINS(name,value,old)			\
	(((old) & ~(((1 << USART_##name##_SIZE) - 1)	\
		    << USART_##name##_OFFSET))		\
	 | USART_BF(name,value))

/* Register access macros */
#define usart_read(regs, reg)						\
	(((volatile struct usart_regs *)regs)->reg)
#define usart_write(regs, reg, value)					\
	(((volatile struct usart_regs *)regs)->reg = (value))

#endif /* USART_REGS_H_INCLUDED */
