/**
 * \file
 *
 * \brief Serial console backend.
 *
 * This is the interface to a very simple serial driver serving as a
 * console backend driver.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#ifndef CONSOLE_SERIAL_H_INCLUDED
#define CONSOLE_SERIAL_H_INCLUDED

#include <status-codes.h>
#include <chip/usart.h>

/**
 * \internal
 * Undefined function which will cause a linker error if
 * serial_console_init() is called with a constant and invalid serial
 * port ID.
 */
extern void __serial_console_bad_constant_port(unsigned int port);

extern int __serial_console_init(struct console *con, unsigned int port,
		void *regs, unsigned long baud_rate, unsigned long flags);

#define SERIAL_ENABLE_TX	(1 << 0)	//!< Enable USART TX
#define SERIAL_ENABLE_RX	(1 << 1)	//!< Enable USART RX

/**
 * \brief Initialize a serial console.
 *
 * \param con The console to be initialized
 * \param port The serial port ID to use
 * \param baud_rate The initial baud rate of the serial port
 * \param flags Bitmask of flags specifying additional configuration
 * parameters.
 *
 * \return 0 on success, or a negative error code on failure.
 *
 * \note Defined partially inline because port and flags will usually
 * be constant.
 */
static inline int serial_console_init(struct console *con, unsigned int port,
		unsigned long baud_rate, unsigned long flags)
{
	void		*regs;
	int		ret;

	regs = get_usart_regs(port);
	if (!regs) {
		if (is_constant(port))
			__serial_console_bad_constant_port(port);
		return -STATUS_INVALID_PARAM;
	}

	ret = __serial_console_init(con, port, regs, baud_rate, flags);
	if (ret)
		return ret;

	return 0;
}

#endif /* CONSOLE_SERIAL_H_INCLUDED */
