/**
 * \file
 *
 * Copyright (C) 2009 Atmel Corporation. All rights reserved.
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/*!
 * \file
 *
 * \brief Byte order conversion helpers
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Copyright (C) 2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef BYTEORDER_H_INCLUDED
#define BYTEORDER_H_INCLUDED

#include <compiler.h>
#include <types.h>

#include <arch/byteorder.h>

#ifndef swab32
/** \brief Swap the order of bytes in a 32-bit word */
# define swab32(x)					\
	((((x) & 0xff) << 24) | (((x) & 0xff00) << 8)	\
	 | (((x) >> 8) & 0xff00) | (((x) >> 24) & 0xff))
#endif
#ifndef swab16
# define swab16(x)	((((x) & 0xff) << 8) | (((x) >> 8) & 0xff))
#endif

#ifdef CPU_IS_BIG_ENDIAN

#define LE16(x)		((le16_t __force)((((x) & 0xff) << 8)		\
					| (((x) >> 8) & 0xff)))
#define LE32(x)		((le32_t __force)((((x) & 0xff) << 24)		\
					| (((x) & 0xff00) << 8)		\
					| (((x) >> 8) & 0xff00)		\
					| (((x) >> 24) & 0xff)))
#define BE16(x)		((be16_t __force)(x))
#define BE32(x)		((be32_t __force)(x))

static inline uint16_t le16_to_cpu(le16_t x)
{
	return swab16((uint16_t __force)x);
}
static inline uint32_t le32_to_cpu(le32_t x)
{
	return swab32((uint32_t __force)x);
}
static inline le16_t cpu_to_le16(uint16_t x)
{
	return (le16_t __force)swab16(x);
}
static inline le32_t cpu_to_le32(uint32_t x)
{
	return (le32_t __force)swab32(x);
}

static inline uint16_t be16_to_cpu(be16_t x)
{
	return (uint16_t __force)x;
}
static inline uint32_t be32_to_cpu(be32_t x)
{
	return (uint32_t __force)x;
}
static inline be16_t cpu_to_be16(uint16_t x)
{
	return (be16_t __force)x;
}
static inline be32_t cpu_to_be32(uint32_t x)
{
	return (be32_t __force)x;
}

#else /* BIG_ENDIAN */

#define LE16(x)		((le16_t __force)(x))
#define LE32(x)		((le32_t __force)(x))
#define BE16(x)		((be16_t __force)((((x) & 0xff) << 8)		\
					| (((x) >> 8) & 0xff)))
#define BE32(x)		((be32_t __force)((((x) & 0xff) << 24)		\
					| (((x) & 0xff00) << 8)		\
					| (((x) >> 8) & 0xff00)		\
					| (((x) >> 24) & 0xff)))

static inline uint16_t le16_to_cpu(le16_t x)
{
	return (uint16_t __force)x;
}
static inline uint32_t le32_to_cpu(le32_t x)
{
	return (uint32_t __force)x;
}
static inline le16_t cpu_to_le16(uint16_t x)
{
	return (le16_t __force)x;
}
static inline le32_t cpu_to_le32(uint32_t x)
{
	return (le32_t __force)x;
}

static inline uint16_t be16_to_cpu(be16_t x)
{
	return swab16((uint16_t __force)x);
}
static inline uint32_t be32_to_cpu(be32_t x)
{
	return swab32((uint32_t __force)x);
}
static inline be16_t cpu_to_be16(uint16_t x)
{
	return (be16_t __force)swab16(x);
}
static inline be32_t cpu_to_be32(uint32_t x)
{
	return (be32_t __force)swab32(x);
}

#endif /* BIG_ENDIAN */

#endif /* BYTEORDER_H_INCLUDED */
