/**
 * \file
 *
 * Copyright (C) 2009 Atmel Corporation. All rights reserved.
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/*!
 * \file
 *
 * \brief Generic Buffer structure and associated helper functions.
 *
 * This is a generic buffer data structure used by several subsystems.
 * Chains of buffers may be passed around between different drivers,
 * minimizing the cost of copying and data conversion.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices with a USB host and/or device controller
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Copyright (C) 2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef BUFFER_H_INCLUDED
#define BUFFER_H_INCLUDED

#include <dma.h>
#include <slist.h>
#include <types.h>

/**
 * \brief A generic data buffer
 *
 * This buffer represents a physically and virtually contiguous memory
 * area used for data storage.
 */
struct buffer {
	/** DMA descriptor associated with this buffer, if any. */
	void			*dma_desc;
	/** Address of the data area */
	buffer_addr_t		addr;
	/** Length of the data area in bytes. */
	size_t			len;
	/** List node in the containing request object's buffer list */
	struct slist_node	node;
};

/**
 * \brief Initialize a buffer for transmitting data
 * \param buf The buffer
 * \param data Data area to associate with the buffer
 * \param len Length of the data area in bytes
 */
static inline void buffer_init_tx(struct buffer *buf,
		const void *data, size_t len)
{
	buf->addr.ptr = (void *)data;
	buf->addr.phys = dma_map_single(buf->addr.ptr, len, DMA_TO_DEVICE);
	buf->len = len;
}

/**
 * \brief Initialize an already-DMA-mapped buffer for transmitting
 * data
 *
 * \param buf The buffer
 * \param data Virtual address of data area to associate with the buffer
 * \param phys Physical address of data area
 * \param len Length of the data area in bytes
 */
static inline void buffer_init_tx_mapped(struct buffer *buf,
		const void *data, phys_addr_t phys, size_t len)
{
	buf->addr.ptr = (void *)data;
	buf->addr.phys = phys;
	buf->len = len;
}

/**
 * \brief Initialize a buffer for receiving data
 * \param buf The buffer
 * \param data Data area to associate with the buffer
 * \param len Length of the data area in bytes
 */
static inline void buffer_init_rx(struct buffer *buf, void *data, size_t len)
{
	buf->addr.ptr = data;
	buf->addr.phys = dma_map_single(data, len, DMA_FROM_DEVICE);
	buf->len = len;
}

/**
 * \brief Initialize an already-DMA-mapped buffer for receiving data
 *
 * \param buf The buffer
 * \param data Virtual address of data area to associate with the buffer
 * \param phys Physical address of data area
 * \param len Length of the data area in bytes
 */
static inline void buffer_init_rx_mapped(struct buffer *buf,
		void *data, phys_addr_t phys, size_t len)
{
	buf->addr.ptr = data;
	buf->addr.phys = phys;
	buf->len = len;
}

extern struct buffer *buffer_alloc(void);
extern void buffer_free(struct buffer *buf);
extern void buffer_resize(struct buffer *buf, size_t new_size);

extern struct buffer *buffer_dma_alloc(size_t size);
extern void buffer_dma_free(struct buffer *buf, size_t size);

extern void buffer_pool_init(void);

#endif /* BUFFER_H_INCLUDED */
