/**
 * \file
 *
 * \brief Interface for block copy between block devices
 *
 * This file contains the interface used to copy a chunk of blocks
 * from one block device to another.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef BLKDEV_COPY_H_INCLUDED
#define BLKDEV_COPY_H_INCLUDED

#include <workqueue.h>
#include <block/device.h>

/** \brief Error code reported if necessary memory could not be allocated */
#define BLOCK_COPY_NOMEM		1
/** \brief Error code reported if the block size of the devices do not match */
#define BLOCK_COPY_BLOCK_SIZE_MISMATCH	2

/**
 * Structure holding a description of a copy operation.
 */
struct block_copy_request {
	/** \brief Source of the copy operation */
	struct block_device *source;
	/** \brief Target of the copy operation */
	struct block_device *target;
	/** \brief First block of the source to copy from */
	block_addr_t source_start;
	/** \brief First block of the target to copy to */
	block_addr_t target_start;
	/** \brief Number of blocks to copy */
	uint32_t length;
	/** \brief Number of blocks copied so far (passed to callbacks) */
	uint32_t total_blocks_copied;
	/** \brief Status of the copy operation */
	int status;
	/** \brief Status of the copy operation on source */
	int source_status;
	/** \brief Status of the copy operation on destination */
	int dest_status;
	/** \brief Callback called when the operation is completed */
	void (*completion_callback)(struct block_copy_request *req,
			uint32_t blocks_copied, void *data);
	/** \brief Callback called after each successful write operation */
	void (*progress_callback)(struct block_copy_request *req,
			uint32_t blocks_copied, void *data);
	/** \brief Data passed to the callback */
	void *data;
	/** \brief Block request used when reading from source */
	struct block_request *read_req;
	/** \brief Block request used when writing to target */
	struct block_request *write_req;
	/** \brief List of buffers used for the transfer */
	struct slist buffer_list;
	/** \brief Total number of blocks allocated as buffers */
	uint32_t blocks_allocated;
	/** \brief Block size of the devices (must match) */
	size_t block_size;
	/** \brief Numbers of blocks that fit into one buffer */
	size_t blocks_pr_buf;
	/** \brief Workqueue item used when adding work to the workqueue */
	struct workqueue_item work;
};

struct block_copy_request *block_copy_alloc(
		uint32_t nr_blocks,
		uint32_t block_size,
		void *data);

void block_copy_free(struct block_copy_request *request);

int block_copy_set_source(
		struct block_copy_request *req,
		struct block_device *dev,
		block_addr_t lba);

int block_copy_set_dest(
		struct block_copy_request *req,
		struct block_device *dev,
		block_addr_t lba);

void block_copy_set_completion_callback(
		struct block_copy_request *req,
		void (*callback)
			(struct block_copy_request *, uint32_t, void *));

void block_copy_set_progress_callback(
		struct block_copy_request *req,
		void (*callback)
			(struct block_copy_request *, uint32_t, void *));

void block_copy_submit(struct block_copy_request *req);

#endif /* BLKDEV_COPY_H_INCLUDED */
