/**
 * \file
 *
 * Copyright (C) 2009 Atmel Corporation. All rights reserved.
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/*!
 * \file
 *
 * \brief Generic bit operations
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Copyright (C) 2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef BITOPS_H_INCLUDED
#define BITOPS_H_INCLUDED

#include <stdbool.h>
#include <types.h>
#include <assert.h>

/** \brief Generate a bit mask with only bit \a nr set */
#define bit_mask(nr)	(1 << ((nr) & (BITS_PER_LONG - 1)))
/**
 * \brief Return the offset of the word containing bit \a nr in a
 * multi-word bitfield.
 */
#define bit_word(nr)	((nr) / BITS_PER_LONG)

#include <arch/bitops.h>

/**
 * \brief Set bit \a nr in the multi-word bitfield \a bitmap
 */
static inline void set_bit(unsigned int nr, unsigned long *bitmap)
{
	bitmap[bit_word(nr)] |= bit_mask(nr);
}

/**
 * \brief Clear bit \a nr in the multi-word bitfield \a bitmap
 */
static inline void clear_bit(unsigned int nr, unsigned long *bitmap)
{
	bitmap[bit_word(nr)] &= ~bit_mask(nr);
}

/**
 * \brief Toggle bit \a nr in the multi-word bitfield \a bitmap
 */
static inline void toggle_bit(unsigned int nr, unsigned long *bitmap)
{
	bitmap[bit_word(nr)] ^= bit_mask(nr);
}

/**
 * \brief Test bit \a nr in the multi-word bitfield \a bitmap
 *
 * \retval true if the bit is set
 * \retval false if the bit is clear
 */
static inline bool test_bit(unsigned int nr, unsigned long *bitmap)
{
	return 1UL & (bitmap[bit_word(nr)] >> (nr & (BITS_PER_LONG - 1)));
}

#ifndef extract_aligned_bitfield
/**
 * \brief Get the bitfield of \a length starting at bit nr \a start in
 * the multi-word bitfield \a bitmap
 *
 * \pre The desired bitfield must not cross word boundaries
 *
 * \retval The desired bitfield
 */
static inline uint32_t extract_aligned_bitfield(
		unsigned int start,
		unsigned int length,
		const uint32_t *bitmap)
{
	uint32_t tmp;

	tmp = bitmap[bit_word(start)] >> (start & (BITS_PER_LONG - 1));
	return tmp & ((1 << length) - 1);
}
#endif

/**
 * \brief Get the bitfield of \a length starting at bit nr \a start in
 * the multi-word bitfield \a bitmap
 *
 * \retval The desired bitfield
 */
static inline uint32_t extract_bitfield(
		unsigned int start,
		unsigned int length,
		const uint32_t *bitmap)
{
	assert(length <= BITS_PER_LONG);

	if (bit_word(start) == bit_word(start + length)) {
		return extract_aligned_bitfield(start, length, bitmap);
	} else {
		unsigned int length_first;
		uint32_t bf_first;
		uint32_t bf_second;

		length_first = BITS_PER_LONG - (start & (BITS_PER_LONG - 1));
		bf_first = extract_aligned_bitfield(start,
				length_first, bitmap);
		bf_second = extract_aligned_bitfield(start + length_first,
				length - length_first, bitmap);

		return bf_first | (bf_second << length_first);
	}
}

#endif /* BITOPS_H_INCLUDED */
