/**
 * \file
 *
 * \brief Interface of Asynchronous Advanced Encryption Standard (AES) driver
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef AES_H_INCLUDED
#define AES_H_INCLUDED

#include <stdint.h>
#include <stdbool.h>
#include <compiler.h>
#include <dmac/dma_controller.h>

#define AES_DECRYPT		0 <<  0	//!< Set AES mode to decrypt
#define AES_ENCRYPT		1 <<  0 //!< Set AES mode to encrypt
#define AES_KEYSIZE_128		0 << 10	//!< Use 128bit key
#define AES_KEYSIZE_192		1 << 10	//!< Use 192bit key
#define AES_KEYSIZE_256		2 << 10 //!< Use 256bit key
#define AES_KEYSIZE_MASK	3 << 10 //!< Keysize mask
#define AES_OPMODE_ECB		0 << 12 //!< Set AES operation mode to ECB
#define AES_OPMODE_CBC		1 << 12 //!< Set AES operation mode to CBC
#define AES_OPMODE_OFB		2 << 12 //!< Set AES operation mode to OFB
#define AES_OPMODE_CFB		3 << 12 //!< Set AES operation mode to CFB
#define AES_OPMODE_CTR		4 << 12 //!< Set AES operation mode to CTR

/** \brief AES module struct */
struct aes_module {
	void *port;
	uint32_t mode;

	struct dmac_channel *dma_tx_channel;
	struct dmac_channel *dma_rx_channel;

	int counter;
};

/** \brief AES request struct */
struct aes_request {
	struct aes_module *module;
	struct slist tx_buf_list;
	struct slist rx_buf_list;
	struct dmac_request tx_req;
	struct dmac_request rx_req;
	/** \brief Request done callback */
	void (*req_done)(struct aes_request *, struct slist *, void *);
	/** \brief Arbitrary pointer passed to the callback */
	void *context;
	int done_counter;
};

void aes_init(struct aes_module *module, uint32_t mode);
void aes_free(struct aes_module *module);
void aes_load_key(struct aes_module *module, const uint32_t *key);
void aes_set_cipher(struct aes_module *module, uint32_t mode);
uint32_t aes_get_cipher(struct aes_module *module);
void aes_load_init_vector(struct aes_module *module, const uint32_t *key);

void aes_submit_request(struct aes_request *req);
int aes_duplicate_buffer_refs(struct slist *copy, struct slist *original);
void aes_free_duplicate_buffers(struct slist *buf_list);
struct aes_request *aes_alloc_request(struct aes_module *module);
void aes_free_request(struct aes_request *req);
void aes_prepare_request(struct aes_request *req, struct slist *buf_list);

#endif  /* AES_H_INCLUDED */
