/**
 * \file
 *
 * \brief Common LED helper functions
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#include <led.h>
#include <softirq.h>
#include <app/timer.h>

static void led_blink_toggle(void *data)
{
	struct blinking_led	*blink = data;

	if (!blink->enabled) {
		blink->active = false;
		return;
	}

	if (blink->state) {
		blink->state = false;
		led_deactivate(blink->led);
	} else {
		blink->state = true;
		led_activate(blink->led);
	}

	timer_add_task_us(&led_timer, &blink->task,
			blink->period_ms * 1000);
}

/**
 * \brief Start blinking a LED
 * \param blink LED blinking state
 */
void led_blink_start(struct blinking_led *blink)
{
	softirq_disable();
	if (!blink->enabled) {
		blink->enabled = true;
		if (!blink->active) {
			blink->state = false;
			led_deactivate(blink->led);
			timer_add_task_us(&led_timer, &blink->task,
					blink->period_ms * 1000);
		}
	}
	softirq_enable();
}

/**
 * \brief Stop blinking a LED
 * \param blink LED blinking state
 */
void led_blink_stop(struct blinking_led *blink)
{
	softirq_disable();
	blink->enabled = false;
	softirq_enable();
}

/**
 * \brief Initialize LED blinking state
 *
 * \param blink LED blinking state to be initialized
 * \param led The ID of the blinking LED
 * \param period_ms Time between each toggle in milliseconds
 */
void led_blink_init(struct blinking_led *blink, enum board_led_id led,
		unsigned int period_ms)
{
	blink->led = led;
	blink->period_ms = period_ms;
	timer_init_task(&blink->task, led_blink_toggle, blink);
	blink->enabled = false;
	blink->active = false;
}
