/**
 * \file
 *
 * \brief GPIO irq handler
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a GPIO module
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#include <debug.h>
#include <util.h>
#include <slist.h>
#include <interrupt.h>
#include <chip/portmux.h>
#include <gpio/gpio_irq_handler.h>
#include <chip/gpio_irq_group.h>

/**
 * \brief GPIO irq handler structure
 */
struct gpio_irq_group {
	struct slist	handler_list;	/**< List of handlers */
};

void gpio_priv_irq_group_interrupt(void *data)
{
	struct gpio_irq_group		*irqg = data;
	struct gpio_irq_handler		*irqh;

	slist_for_each(&irqg->handler_list, irqh, node) {
		if (gpio_int_is_pending(irqh->pin)) {
			gpio_clear_int_flag(irqh->pin);
			irqh->handler(irqh->data);
		}
	}
}

static void gpio_init_irq_groups(struct gpio_irq_group *irqg)
{
	int i;

	for (i = 0; i < GPIO_IRQ_NR_GROUPS; i++) {
		slist_init(&irqg[i].handler_list);
		gpio_setup_irq_group(i, &irqg[i]);
	}
}

void gpio_register_irq_handler(struct gpio_irq_handler *irqh)
{
	unsigned int flags;
	int index;
	static bool initialized = false;
	static struct gpio_irq_group irqg[GPIO_IRQ_NR_GROUPS];

	flags = cpu_irq_save();
	if (!initialized) {
		initialized = true;
		gpio_init_irq_groups(irqg);
	}

	index = gpio_irq_group_pin2index(irqh->pin);
	slist_insert_tail(&irqg[index].handler_list, &irqh->node);
	cpu_irq_restore(flags);
}

