/**
 * \file
 *
 * \brief Debug console implementation
 *
 * This file implements a simple debug console which can be used to
 * dump information over a serial line when debugging a new
 * application or driver.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#include <debug.h>
#include <console/core.h>
#include <console/serial.h>

/**
 * \internal
 * \brief The debug console
 */
static struct console debug_console;

/**
 * \internal
 * \brief Formatted output conversion to the debug console.
 *
 * Produce output according to \a format on the debug console. \a
 * format is interpreted as a regular printf()-style format string
 * with a few limitations (some specifiers are accepted but ignored.)
 *
 * \param format Format specification.
 * \param ap Format arguments.
 *
 * \return The number of characters printed.
 */
int dbg_priv_vprintf(const char *format, va_list ap)
{
	return console_vprintf(&debug_console, format, ap);
}

/**
 * \internal
 * \brief Formatted output conversion to the debug console.
 *
 * Produce output according to \a format on the debug console. \a
 * format is interpreted as a regular printf()-style format string
 * with a few limitations (some specifiers are accepted but ignored.)
 *
 * \param format Format specification.
 *
 * \return The number of characters printed.
 */
int dbg_priv_printf(const char *format, ...)
{
	va_list ap;
	int n;

	va_start(ap, format);
	n = dbg_priv_vprintf(format, ap);
	va_end(ap);

	return n;
}

/**
 * \internal
 * \brief Write a string to the debug console.
 *
 * \param str NUL-terminated string.
 *
 * \return The number of characters written.
 */
int dbg_priv_putstr(const char *str)
{
	return console_putstr(&debug_console, str);
}

/**
 * \internal
 * \brief Write a single character to the debug console.
 *
 * \param c The character to write.
 *
 * \return \a c as an unsigned char cast to an int.
 */
int dbg_priv_putchar(int c)
{
	return console_putchar(&debug_console, c);
}

/**
 * \brief Initialize the debug console.
 *
 * This function must be called before any of the other debug console
 * functions may be called. If the debug console is disabled, this
 * function does nothing.
 */
void dbg_init(void)
{
	serial_console_init(&debug_console, CONFIG_DEBUG_CONSOLE_USART,
			CONFIG_DEBUG_CONSOLE_BAUD_RATE, SERIAL_ENABLE_TX);
}
