/*This file is prepared for Doxygen automatic documentation generation.*/
//! \file *********************************************************************
//!
//! \brief Main for USB application.
//!
//! - Compiler:           IAR EWAVR and GNU GCC for AVR
//! - Supported devices:  AT90USB1287, AT90USB1286, AT90USB647, AT90USB646
//!
//! \author               Atmel Corporation: http://www.atmel.com \n
//!                       Support and FAQ: http://support.atmel.no/
//!
//! ***************************************************************************
//!
//! @mainpage AT90USBxxx Dual role demonstration with file system
//!
//! @section intro License
//! Use of this program is subject to Atmel's End User License Agreement.
//!
//! Please read file  \ref lic_page for copyright notice.
//!
//! @section install Description
//! This embedded application code illustrates how to implement a dual role application with
//! the AT90USBxxx controller.
//!
//! As the AT90USBxxx implements  a device/host usb controller, the embedded application can operates
//! in one of the following usb operating modes:
//! - USB device
//! - USB reduced host controller
//!
//! @section src_code About the source code
//! This source code is usable with the following compilers:
//! - IAR Embedded Workbench (5.11A and higher)
//! - AVRGCC (WinAVR 20080411 and higher).
//!
//! Support for other compilers may required modifications or attention for:
//! - compiler.h file 
//! - special registers declaration file
//! - interrupt subroutines declarations
//!
//! @section sample About the sample application
//! By default the demonstration is preconfigured for dual role USB application.
//! It means that the code generated allows to operate as a device or a host depending on the USB ID pin:
//! - Attached to a mini B plug (ID pin unconnected) the application will be used in the device operating mode.
//! Thus the application can be connected to a system host (PC) to operate with the three modes:
//!   - Usb mouse (using joystick)
//!   - Usb mass storage (on board dataflash))
//!   - Usb generic HID (A pc side applications allow to communicate with the embedded firmware, see \ref UsbHidGenericDemos).
//! - Attached to a mini A plug (ID pin tied to ground) the application will operate in reduced host mode.
//! This mode allows to connect one of the following devices:
//!   - a standard USB mouse
//!   - a USB mass storage device
//!   - an AT90USB controller in USB DFU Bootloader  
//!   - or another application in B device mode. 
//! @image html appli_host.gif
//!
//! @section device Using the USB device mode
//! Connect the STK525 to a PC host system with a miniB (STK525 side) to plugA (PC host) cable
//! - The joystick allows to move the mouse cursor 
//! - A removable drive allows to read/write files on the on-board data flash memory 
//! (for example to read write the "IN", "OUT" directories and "FIRMWARE.HEX" mentionned in the section \ref host_usage )
//! - The ATMEL hid AT90USBxxx pc side application allows to read temperature, turn ON/OFF leds
//! and get joystick events 
//!
//! @section host_usage Using the USB host mode
//! Connected  in host mode (miniA plug inserted), the behavior of the application depends on the device type connected:
//! - Connected to a standard USB mouse, the mouse movements are visible with the LEDs.
//! - Connected to a USB Mass Storage Device (MSD), files can be exchange with the  on-board data flash disk and the MSD.
//!   - The right direction of the joystick allows to read the 
//!content of a MSD "OUT" directory and to write it to an on-board data flash "IN" directory.
//!   - The left direction of the joystick allows to write the 
//!content of an on-board data flash "OUT" directory to a MSD "IN" directory.
//!   - For more information about File System module see \subpage FileSystem
//! - Connected to an AT90USB DFU bootloader, the left direction of the joystick allows to download 
//! a new firmware in the AT90USB device connected. The firmware is an Intel Hex file named "FIRMWARE.HEX" located to the 
//! root directory of the on-board data flash drive. 
//!
//!
//! @section arch Architecture
//! As illustrated in the figure bellow, the application entry point is located is the main.c file.
//! The main function first performs the initialization of a scheduler module and then runs it in an infinite loop.
//! The scheduler is a simple infinite loop calling all its tasks defined in the conf_scheduler.h file.
//! No real time schedule is performed, when a task ends, the scheduler calls the next task defined in
//! the configuration file (conf_scheduler.h).
//!
//! The sample dual role application is based on several different tasks:
//! - The usb_task  (usb_task.c associated source file), is the task performing the USB low level
//! enumeration process in device or host mode.
//! Once this task has detected that the usb connection is fully operationnal, it updates different status flags
//! that can be check within the high level application tasks.
//! - The storage task performs SCSI bulk only protocol decoding and performs flash memory access.
//! - The hid task performs the high level device application operation.
//! Once the device is fully enumerated (DEVICE SETUP_SET_CONFIGURATION request received), the task
//! checks for received data on its OUT endpoint and transmit  data on its IN endpoint.
//! - The mouse task performs mouse events decoding and report construction to the USB host.
//! - The host mouse task manage events received from the USB mouse connected to display mouse activity on LEDs
//! - The host_ms_task manages the Device Mass Storage interface connected
//!   When joy is right (or left), the task load the IN directory from device to OUT of DataFlash (from DataFlash to device)
//! - The host_dfu_task manages the AT90USB DFU bootloader connection.
//!   When joy is up and left, the task load the FIRMWARE.HEX file from DataFlash to DFU Device.
//!
//! \image html arch_full.gif
//!
//! ***************************************************************************

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
//_____  I N C L U D E S ___________________________________________________

#include "config.h"
#include "modules/scheduler/scheduler.h"
#include "lib_mcu/wdt/wdt_drv.h"
#include "lib_mcu/power/power_drv.h"
#include "lib_mcu/util/start_boot.h"

//_____ M A C R O S ________________________________________________________

//_____ D E F I N I T I O N S ______________________________________________

int main(void)
{
   start_boot_if_required();
   wdtdrv_disable();
   Clear_prescaler();
   scheduler();
   return 0;
}

//! \name Procedure to speed up the startup code
//! This one increment the CPU clock before RAM initialisation
//! @{
#ifdef  __GNUC__
// Locate low level init function before RAM init (init3 section)
// and remove std prologue/epilogue
char __low_level_init(void) __attribute__ ((section (".init3"),naked));
#endif

#ifdef __cplusplus
extern "C" {
#endif
char __low_level_init()
{
  Clear_prescaler();
  return 1;
}
#ifdef __cplusplus
}
#endif
//! @}

