/*This file is prepared for Doxygen automatic documentation generation.*/
//! \file *********************************************************************
//!
//! \brief This file contains the low level macros and definition for stk600 board
//!
//! - Compiler:           IAR EWAVR and GNU GCC for AVR
//! - Supported devices:  AT90USB1287, AT90USB1286, AT90USB647, AT90USB646
//!
//! \author               Atmel Corporation: http://www.atmel.com \n
//!                       Support and FAQ: http://support.atmel.no/
//!
//! ***************************************************************************

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef STK_600_H
#define STK_600_H

//_____ I N C L U D E S ____________________________________________________
#include "config.h"

#if (TARGET_BOARD==STK600)

//_____ M A C R O S ________________________________________________________

//! @defgroup STK_600_module STK600
//! This module contains low level hardware abstraction layer for STK600 board with AT90USB128.
//! Routing card: STK600-RC064U-17
//! Socket card: STK600-TQFP64
//!
//! The STK600 software configuration (use AVRStudio) :
//! - HW Settings/Clock Generator/8000MHz
//!
//! The STK600 hardware configuration :
//! - set clock switch on EXT clock (to use Clock generator from STK600)
//! - connect PORTB to SWITCHES
//! - connect PORTD to LEDS
//!
//! This configuration enables following interface switches 0 to 7 and leds 0 and 7.
//! @image html stk_600.jpg
//! @{


      //! @defgroup STK_600_leds Leds Management
      //! Macros to manage Leds on stk600
      //! @{
#define  LED_PORT             PORTD
#define  LED_DDR              DDRD
#define  LED_PIN              PIND

#define  LED0_BIT             PIND0
#define  LED1_BIT             PIND1
#define  LED2_BIT             PIND2
#define  LED3_BIT             PIND3
#define  LED4_BIT             PIND4
#define  LED5_BIT             PIND5
#define  LED6_BIT             PIND6
#define  LED7_BIT             PIND7                      

#define  Leds_init()          (LED_DDR  |=  (1<<LED0_BIT) | (1<<LED1_BIT) | (1<<LED2_BIT) | (1<<LED3_BIT) | (1<<LED4_BIT) | (1<<LED5_BIT) | (1<<LED6_BIT) | (1<<LED7_BIT),Leds_off())
#define  Leds_on()            (LED_PORT &= ~((1<<LED0_BIT) | (1<<LED1_BIT) | (1<<LED2_BIT) | (1<<LED3_BIT) | (1<<LED4_BIT) | (1<<LED5_BIT) | (1<<LED6_BIT) | (1<<LED7_BIT)))
#define  Leds_off()           (LED_PORT |=  (1<<LED0_BIT) | (1<<LED1_BIT) | (1<<LED2_BIT) | (1<<LED3_BIT) | (1<<LED4_BIT) | (1<<LED5_BIT) | (1<<LED6_BIT) | (1<<LED7_BIT))
#define  Leds_set_val(val)    (Leds_off(),LED_PORT |= val&((1<<LED0_BIT) | (1<<LED1_BIT) | (1<<LED2_BIT) | (1<<LED3_BIT) | (1<<LED4_BIT) | (1<<LED5_BIT) | (1<<LED6_BIT) | (1<<LED7_BIT)))
#define  Leds_get_val()       (LED_PORT)

#define  Led0_on()            (LED_PORT &= ~(1<<LED0_BIT))
#define  Led1_on()            (LED_PORT &= ~(1<<LED1_BIT))
#define  Led2_on()            (LED_PORT &= ~(1<<LED2_BIT))
#define  Led3_on()            (LED_PORT &= ~(1<<LED3_BIT))
#define  Led4_on()            (LED_PORT &= ~(1<<LED4_BIT))
#define  Led5_on()            (LED_PORT &= ~(1<<LED5_BIT))
#define  Led6_on()            (LED_PORT &= ~(1<<LED6_BIT))
#define  Led7_on()            (LED_PORT &= ~(1<<LED7_BIT))
#define  Led0_off()           (LED_PORT |=  (1<<LED0_BIT))
#define  Led1_off()           (LED_PORT |=  (1<<LED1_BIT))
#define  Led2_off()           (LED_PORT |=  (1<<LED2_BIT))
#define  Led3_off()           (LED_PORT |=  (1<<LED3_BIT))
#define  Led4_off()           (LED_PORT |=  (1<<LED4_BIT))
#define  Led5_off()           (LED_PORT |=  (1<<LED5_BIT))
#define  Led6_off()           (LED_PORT |=  (1<<LED6_BIT))
#define  Led7_off()           (LED_PORT |=  (1<<LED7_BIT))
#define  Led0_toggle()        (LED_PIN  |=  (1<<LED0_BIT))
#define  Led1_toggle()        (LED_PIN  |=  (1<<LED1_BIT))
#define  Led2_toggle()        (LED_PIN  |=  (1<<LED2_BIT))
#define  Led3_toggle()        (LED_PIN  |=  (1<<LED3_BIT))
#define  Led4_toggle()        (LED_PIN  |=  (1<<LED4_BIT))
#define  Led5_toggle()        (LED_PIN  |=  (1<<LED5_BIT))
#define  Led6_toggle()        (LED_PIN  |=  (1<<LED6_BIT))
#define  Led7_toggle()        (LED_PIN  |=  (1<<LED7_BIT))
#define  Is_led0_on()         (LED_PIN  &   (1<<LED0_BIT) ? FALSE : TRUE)
#define  Is_led1_on()         (LED_PIN  &   (1<<LED1_BIT) ? FALSE : TRUE)
#define  Is_led2_on()         (LED_PIN  &   (1<<LED2_BIT) ? FALSE : TRUE)
#define  Is_led3_on()         (LED_PIN  &   (1<<LED3_BIT) ? FALSE : TRUE)
#define  Is_led4_on()         (LED_PIN  &   (1<<LED4_BIT) ? FALSE : TRUE)
#define  Is_led5_on()         (LED_PIN  &   (1<<LED5_BIT) ? FALSE : TRUE)
#define  Is_led6_on()         (LED_PIN  &   (1<<LED6_BIT) ? FALSE : TRUE)
#define  Is_led7_on()         (LED_PIN  &   (1<<LED7_BIT) ? FALSE : TRUE)

      //! @}

      //! @defgroup STK_600_joy Joystick Management
      //! Macros to manage Joystick on stk600
      //! 
      //! The switchs are associated at a key name like STK525 or USB KEY :
      //! - Switch 6 associated to left bouton
      //! - Switch 5 associated to center bouton
      //! - Switch 4 associated to right bouton
      //! - Switch 3 associated to left move
      //! - Switch 2 associated to right move
      //! - Switch 1 associated to up move
      //! - Switch 0 associated to down move
      //!@{
#define  Switches_init()      (DDRB &= ~0xFF, PORTB |= 0xFF)
#define  Is_switch_7()        ((PINB & (1<<PINB7)) ?  FALSE : TRUE)
#define  Is_switch_6()        ((PINB & (1<<PINB6)) ?  FALSE : TRUE)
#define  Is_switch_5()        ((PINB & (1<<PINB5)) ?  FALSE : TRUE)
#define  Is_switch_4()        ((PINB & (1<<PINB4)) ?  FALSE : TRUE)
#define  Is_switch_3()        ((PINB & (1<<PINB3)) ?  FALSE : TRUE)
#define  Is_switch_2()        ((PINB & (1<<PINB2)) ?  FALSE : TRUE)
#define  Is_switch_1()        ((PINB & (1<<PINB1)) ?  FALSE : TRUE)
#define  Is_switch_0()        ((PINB & (1<<PINB0)) ?  FALSE : TRUE)
#define  Is_not_switch_7()    ((PINB & (1<<PINB7)) ?  TRUE : FALSE)
#define  Is_not_switch_6()    ((PINB & (1<<PINB6)) ?  TRUE : FALSE)
#define  Is_not_switch_5()    ((PINB & (1<<PINB5)) ?  TRUE : FALSE)
#define  Is_not_switch_4()    ((PINB & (1<<PINB4)) ?  TRUE : FALSE)
#define  Is_not_switch_3()    ((PINB & (1<<PINB3)) ?  TRUE : FALSE)
#define  Is_not_switch_2()    ((PINB & (1<<PINB2)) ?  TRUE : FALSE)
#define  Is_not_switch_1()    ((PINB & (1<<PINB1)) ?  TRUE : FALSE)
#define  Is_not_switch_0()    ((PINB & (1<<PINB0)) ?  TRUE : FALSE)

#define  Joy_init()           Switches_init()
#define  Is_btn_left()        Is_switch_6()
#define  Is_btn_middle()      Is_switch_5()
#define  Is_btn_right()       Is_switch_4()
#define  Is_joy_left()        Is_switch_3()
#define  Is_joy_right()       Is_switch_2()
#define  Is_joy_up()          Is_switch_1()
#define  Is_joy_down()        Is_switch_0()


#define  Is_btn_not_left()    Is_not_switch_6()
#define  Is_btn_not_middle()  Is_not_switch_5()
#define  Is_btn_not_right()   Is_not_switch_4()
#define  Is_joy_not_left()    Is_not_switch_3()
#define  Is_joy_not_right()   Is_not_switch_2()
#define  Is_joy_not_up()      Is_not_switch_1()
#define  Is_joy_not_down()    Is_not_switch_0()


//! Enable interrupts for switches (PINB0 to PINB7)
//! - Set up interrupts for switches
//! - clear flag
//! - enable interrupt
#define  Switches_enable_it() {  \
      PCMSK0 |= (1<<PCINT0)|(1<<PCINT1)|(1<<PCINT2)|(1<<PCINT3)|(1<<PCINT4)|(1<<PCINT5)|(1<<PCINT6)|(1<<PCINT7);\
      PCIFR |= (1<<PCIF0);       \
      PCICR |= (1<<PCIE0);}

//! Disable interrupts for switches (PINB0 to PINB7)
//! - disable interrupt
//! - clear flag
//! - Clear mask
#define  Switches_disable_it() { \
      PCICR  &= ~(1<<PCIE0);     \
      PCIFR  |= (1<<PCIF0);      \
      PCMSK0 &= ~((1<<PCINT0)|(1<<PCINT1)|(1<<PCINT2)|(1<<PCINT3)|(1<<PCINT4)|(1<<PCINT5)|(1<<PCINT6)|(1<<PCINT7));}
      //! @}


      //! @defgroup STK_600_HWB HWB button management
      //! HWB button is not connected
      //! @{
#define  Hwb_button_init()
#define  Is_hwb()             FALSE
#define  Is_not_hwb()         TRUE
      //! @}

//! @}

#endif   // TARGET_BOARD==STK600

#endif  // STK_600_H
