#include <stdio.h>
#include <string.h>
#include "mti.h"

#define STRING2INT(str, int) sscanf(str, "%d", &int);

#define MAX_INPUT_LINE_SIZE 256
#define MAX_PORT_WIDTH 64
#define MAX_NUM_PORTS 64
#define DRIVE 0
#define TEST  1

/* This is the structure that contains the tester's port data */
typedef struct {
	int   number;		/* unique number for each port */
	char *name;
	int is_array_type;	/* true for arrays, false otherwise */
	int width;
} portstruct;
static portstruct tester_ports[MAX_NUM_PORTS];

/* The test data is stored in a linked list whose nodes are
	described below. The list is terminated with a NULL */
typedef struct tp {
	int portnum;
	int type;		/* either DRIVE or TEST */
	char *test_val;	/* either the value to drive out or the value
						to test for at the given time. */
	struct tp *nxt;
} testpoint;

static testpoint *testpoints = NULL;
static int num_ports = 0;

typedef struct {
	signalID ports[MAX_NUM_PORTS];
	driverID drivers[MAX_NUM_PORTS];
	processID test_values;
	FILE *vectorfile_id;
	int verbose;
} inst_rec, *inst_rec_ptr;

/* this function frees all nodes of the list of testpoints */
static delete_testpoints()
{
	testpoint *tp, *p;

	tp=testpoints;
	while(tp != NULL) {
		p=tp->nxt;
		free(tp);
		tp = p;
	}
	testpoints=NULL;
}

/* this function scans the list of ports for a match and
	returns the unique id number for that port */

static int findportnum(portname)
	char *portname;
{
	int i;
	for (i=0; i<num_ports; i++) {
		if (strcmp(portname, tester_ports[i].name) == 0)
			return tester_ports[i].number;
	}
	return -1;
}

/******************************************************
 The following set of routines is used for converting STD_LOGIC
 and STD_LOGIC_VECTORs from their internal representations
 (ordinal positions in the enerated type for std_logic)
 to the character values (X, 0, 1, ...) and vice-versa.
*/

static mvl9_char[] = {
	'U', 'X', '0', '1', 'Z', 'W', 'L', 'H', '-' };

static char convert_enum_to_mvl9_char(enum_bit)
	int enum_bit;
{
		if (enum_bit > 8)
			return '?';
		else
			return mvl9_char[ enum_bit ];
}

static convert_enums_to_mvl9_string(enums, string, len)
	char *enums;
	char *string;
	int len;
{
	int i;
	for (i=0; i<len; i++) {
		string[i] = convert_enum_to_mvl9_char((int) enums[i]);
	}
	string[len] = '\0';
}

static convert_mvl9_string_to_enums(string, enums, len)
	char *string;
	char *enums;
	int len;
{
	int i;
	for (i=0; i<len; i++) {
		switch(string[i]) {
			case 'U': enums[i] = 0; break;
			case 'X': enums[i] = 1; break;
			case '0': enums[i] = 2; break;
			case '1': enums[i] = 3; break;
			case 'Z': enums[i] = 4; break;
			case 'W': enums[i] = 5; break;
			case 'L': enums[i] = 6; break;
			case 'H': enums[i] = 7; break;
			case '-': enums[i] = 8; break;
			default:
				mti_PrintMessage("error: a non-MVL9 character was found in a signal value specification.\n");
		}
	}
}

/*****************************************************
 This procedure is called by the simulator as a result of an
 mti_ScheduleWakeup() call. Its purpose is to process all drive/test
 data specified for the current timestep. Its schedules drivers for
 drive points by calling mti_ScheduleDriver(). For test points, it
 reads the port values by calling either mti_GetSignalValue or
 mti_GetArraySignalValue. It then compares the current values with the
 expected values and prints an error message if they don't match.
*/
static test_value_proc(ip)
	inst_rec_ptr ip;
{
	char buf[80];
	char *sig_array_val;
	long now, sigval;
	testpoint *cur_testpoint;
	int portnum, i, width, is_array;
	char expected_val[MAX_PORT_WIDTH];
	char actual_val[MAX_PORT_WIDTH];

	now = mti_Now();
	for (cur_testpoint=testpoints; cur_testpoint; cur_testpoint=cur_testpoint->nxt) {
		portnum = cur_testpoint->portnum;
		width = tester_ports[portnum].width;
		is_array = tester_ports[portnum].is_array_type;
		if (cur_testpoint->type == DRIVE) {
			if (!is_array) {
				if (ip->verbose) {
					printf("TIME %d: drive signal %s with value %c\n",
						now, tester_ports[portnum].name,
						convert_enum_to_mvl9_char(*cur_testpoint->test_val));
				}
				mti_ScheduleDriver(ip->drivers[portnum], (long) *cur_testpoint->test_val, 0, MTI_INERTIAL);
			} else {
				char tmpstring[MAX_PORT_WIDTH];
				convert_enums_to_mvl9_string(cur_testpoint->test_val, tmpstring, width);
				if (ip->verbose) {
					printf("TIME %d: drive signal array %s with value %s\n",
						now, tester_ports[portnum].name, tmpstring);
				}
				mti_ScheduleDriver(ip->drivers[portnum], cur_testpoint->test_val, 0, MTI_INERTIAL);
			}
		} else {
			if (!is_array) {
				char exp, act;
				sigval = mti_GetSignalValue(ip->ports[portnum]);
				exp = convert_enum_to_mvl9_char(*cur_testpoint->test_val);
				act = convert_enum_to_mvl9_char(sigval);
				if (ip->verbose) {
					printf("TIME %d: test signal %s for value %c\n",
						now, tester_ports[portnum].name, exp);
				}
				if (sigval != (long) *cur_testpoint->test_val) {
					sprintf(buf, "Miscompare at time %d, signal %s. Expected \'%c\', Actual \'%c\'\n",
						now, tester_ports[portnum].name, exp, act);
					mti_PrintMessage(buf);
				}
			} else {
				sig_array_val = mti_GetArraySignalValue(ip->ports[portnum], NULL);
				convert_enums_to_mvl9_string(cur_testpoint->test_val, expected_val, width);
				convert_enums_to_mvl9_string(sig_array_val, actual_val, width);
				if (ip->verbose) {
					printf("TIME %d: test signal %s for value %s\n",
						now, tester_ports[portnum].name, expected_val);
				}
				for (i=0; i<width; i++) {
					if (sig_array_val[i] != cur_testpoint->test_val[i]) {
						sprintf(buf, "Miscompare at time %d, signal %s. Expected \"%s\", Actual \"%s\"\n",
							now, tester_ports[portnum].name, expected_val, actual_val);
						mti_PrintMessage(buf);
						break;
					}
				}
			}
		}
	}
	delete_testpoints();
	read_next_statement(ip);
}

static schedule_testpoint(ip, portnum, values, time, type)
	inst_rec_ptr ip;
	int portnum;
	char *values;
	int time;
	int type;
{
	testpoint *this_testpoint;
	static testpoint *last_testpoint;
	int now;

	/* create a new testpoint */
	this_testpoint = (testpoint *) malloc(sizeof(testpoint));
	this_testpoint->type    = type;
	this_testpoint->portnum = portnum;
	this_testpoint->test_val = strdup(values);

	/* link in the new testpoint on the end of the list */
	if (!testpoints) {
		testpoints = this_testpoint;
	} else {
		last_testpoint->nxt = this_testpoint;
	}
	this_testpoint->nxt = NULL;
	last_testpoint = this_testpoint;

	/* schedule a wakeup at this time to check the value then */
	now = mti_Now();
	mti_ScheduleWakeup(ip->test_values, abs(time-now));
}

static int is_drive_or_test_statement(keyword)
	char *keyword;
{
	if ((strcmp(keyword, "drive") == 0)
	  || (strcmp(keyword, "test") == 0))
		return 1;
	return 0;
}

static int is_drive_statement(keyword)
	char *keyword;
{
	if (strcmp(keyword, "drive") == 0)
		return 1;
	return 0;
}

static int open_vector_file(ip)
	inst_rec_ptr ip;
{
	if (ip->vectorfile_id = fopen("vectors", "r")) {
		return 1;
	} else {
		mti_PrintMessage("can't open the file \"vectors\"\n");
		if (ip->verbose) {
			printf("can't open the file \"vectors\"\n");
		}
		return 0;
	}
}

/* this function reads the next statement in the vectors file and
	schedules a wakeup in the future to process the data.
*/
static read_next_statement(ip)
	inst_rec_ptr ip;
{
	char line[MAX_INPUT_LINE_SIZE];
	char *kw, *timestamp, *signame, *sigval;
	int  time, test_type, portnum;
	char values[MAX_PORT_WIDTH];
	char buf[80];
	int  done = 0;

	while (!done) {
		if (fgets(line, MAX_INPUT_LINE_SIZE, ip->vectorfile_id) != NULL) {
			kw = strtok(line, " \t\r\n");
			if (kw && is_drive_or_test_statement(kw)) {
				timestamp = strtok(NULL, " \t\r\n");
				STRING2INT(timestamp, time);
				while ((signame = strtok(NULL, " =\t\r\n")) != NULL) {
					sigval  = strtok(NULL, " \t\r\n");
					portnum = findportnum(signame);
					if (portnum >= 0) {
						convert_mvl9_string_to_enums(sigval, values, tester_ports[portnum].width);
						test_type = is_drive_statement(kw) ? DRIVE : TEST;
						schedule_testpoint(ip, portnum, values, time, test_type);
					} else {
						sprintf(buf, "Can't find port named \"%s\".\n", signame);
						mti_PrintMessage(buf);
					}
				}
				done = 1;
			}
		} else {
			done = 1; /* found end of file */
		}
	}
}

/* this function is called by the simulator to initialize the
	tester module. It makes a data structure of the tester's ports
	(taken from the entity's port list and creates a process which
	will handle all the signal changes.
*/
tester_init(region, param, generics, ports)
	regionID region;
	char *param;
	interface_list *generics;
	interface_list *ports;
{
	inst_rec_ptr ip;
	interface_list *p;
	int i, is_array_type;
	char buf[80];
	extern free();
 
	ip = (inst_rec_ptr) malloc(sizeof(inst_rec));
	mti_AddRestartCB(free, ip);
	/* traverse the list of ports and get port names and widths */
	num_ports = 0;
	for (p=ports; p; p=p->nxt) {
		tester_ports[num_ports].name = p->name;
		is_array_type = (mti_GetTypeKind(p->type) == MTI_TYPE_ARRAY);
		tester_ports[num_ports].is_array_type = is_array_type;
		if (is_array_type) {
			tester_ports[num_ports].width = mti_TickLength(p->type);
		} else {
			tester_ports[num_ports].width = 1;
		}
		ip->ports[   num_ports] = p->u.port;
		ip->drivers[ num_ports] = mti_CreateDriver(p->u.port);
		tester_ports[num_ports].number = num_ports;
		num_ports++;
	}
	ip->test_values = mti_CreateProcess("test", test_value_proc, ip);

	/* check for an optional parameter on the FOREIGN attribute that
		indicates whether or not to display debug messages. */

	if (param && strcmp(param, "verbose") == 0) {
		ip->verbose = 1;
	} else {
		ip->verbose = 0;
	}

	/* open the vector file and process the first test/drive statement */

	if (open_vector_file(ip))
		read_next_statement(ip);
}
