/*This file is prepared for Doxygen automatic documentation generation.*/
//! \file *********************************************************************
//!
//! \brief This file contains the low level macros and definition for evk527 board
//!
//! - Compiler:           IAR EWAVR and GNU GCC for AVR
//! - Supported devices:  ATmega32U4
//!
//! \author               Atmel Corporation: http://www.atmel.com \n
//!                       Support and FAQ: http://support.atmel.no/
//!
//! ***************************************************************************

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef EVK_527_H
#define EVK_527_H

//_____ I N C L U D E S ____________________________________________________
#include "config.h"

#if (TARGET_BOARD==EVK527)

#include "lib_mcu/flash/flash_drv.h"   // Because pin JTAG multiplexage

//_____ M A C R O S ________________________________________________________

//! @defgroup EVK_527_module evk527
//! This module contains low level hardware abstraction layer for evk527 board
//! @image html evk_527.gif
//! @{


      //! @defgroup EVK_527_leds Leds Management
      //! Macros to manage Leds on evk527
      //! @{
#define  LED_PORT       PORTD
#define  LED_DDR        DDRD
#define  LED_PIN        PIND
                    
#define  LED0_BIT       PIND5
#define  LED1_BIT       PIND6
#define  LED2_BIT       PIND7
        
#define  Leds_init()    (LED_DDR  |=  (1<<LED0_BIT) | (1<<LED1_BIT) | (1<<LED2_BIT))
#define  Led0_init()    (LED_DDR  |=  (1<<LED0_BIT))
#define  Led1_init()    (LED_DDR  |=  (1<<LED1_BIT))
#define  Led2_init()    (LED_DDR  |=  (1<<LED2_BIT))
#define  Leds_on()      (LED_PORT |=  (1<<LED0_BIT) | (1<<LED1_BIT) | (1<<LED2_BIT))
#define  Leds_off()     (LED_PORT &= ~((1<<LED0_BIT) | (1<<LED1_BIT) | (1<<LED2_BIT)))
#define  Leds_set_val(val) (Leds_off(),LED_PORT |= (val<<5)&((1<<LED0_BIT) | (1<<LED1_BIT) | (1<<LED2_BIT)))
#define  Leds_get_val()    (LED_PORT>>5)


#define  Led0_on()      (LED_PORT |= (1<<LED0_BIT))
#define  Led1_on()      (LED_PORT |= (1<<LED1_BIT))
#define  Led2_on()      (LED_PORT |= (1<<LED2_BIT))
        
#define  Led0_off()     (LED_PORT &= ~(1<<LED0_BIT))
#define  Led1_off()     (LED_PORT &= ~(1<<LED1_BIT))
#define  Led2_off()     (LED_PORT &= ~(1<<LED2_BIT))
        
#define  Led0_toggle()  (LED_PIN |= (1<<LED0_BIT))
#define  Led1_toggle()  (LED_PIN |= (1<<LED1_BIT))
#define  Led2_toggle()  (LED_PIN |= (1<<LED2_BIT))
        
#define  Is_led0_on()   (LED_PIN & (1<<LED0_BIT) ? TRUE : FALSE)
#define  Is_led1_on()   (LED_PIN & (1<<LED1_BIT) ? TRUE : FALSE)
#define  Is_led2_on()   (LED_PIN & (1<<LED2_BIT) ? TRUE : FALSE)
      //! @}

      //! @defgroup EVK_527_joy Joystick Management
      //! Macros to manage Joystick on evk527
      //! @note The up,left and right joy are mapped on JTAG pins
      //! @{
#define  Joy_init()           {DDRC &= ~0x40; PORTC |= 0x40; if(!Is_JTAG_enable()) {DDRF &= ~0xF0; PORTF |= 0xF0;}}
#define  Is_joy_right()       (Is_JTAG_enable() || (PINF & (1<<PINF7)) ?  FALSE : TRUE)
#define  Is_joy_not_right()   ((!Is_JTAG_enable()) && (PINF & (1<<PINF7)) ?  TRUE : FALSE)
#define  Is_joy_select()      (Is_JTAG_enable() || (PINF & (1<<PINF6)) ?  FALSE : TRUE)
#define  Is_joy_not_select()  ((!Is_JTAG_enable()) && (PINF & (1<<PINF6)) ?  TRUE : FALSE)
#define  Is_joy_up()          (Is_JTAG_enable() || (PINF & (1<<PINF5)) ?  FALSE : TRUE)
#define  Is_joy_not_up()      ((!Is_JTAG_enable()) && (PINF & (1<<PINF5)) ?  TRUE  : FALSE)
#define  Is_joy_left()        (Is_JTAG_enable() || (PINF & (1<<PINF4)) ?  FALSE : TRUE)
#define  Is_joy_not_left()    ((!Is_JTAG_enable()) && (PINF & (1<<PINF4)) ?  TRUE : FALSE)
#define  Is_joy_down()        ((PINC & (1<<PINC6)) ?  FALSE : TRUE)
#define  Is_joy_not_down()    ((PINC & (1<<PINC6)) ?  TRUE : FALSE)

//! Enable interrupts for switches (no possible)
#define  Switches_enable_it()

//! Disable interrupts for switches (no possible)
#define  Switches_disable_it()

      //! @}


      //! @defgroup EVK_527_hwb HWB button management
      //! HWB button is connected to PE2 and can also
      //! be used as generic push button
      //! @{
#define  Hwb_button_init()    (DDRE  &= ~(1<<PINE2), PORTE |= (1<<PINE2))
#define  Is_hwb()             ((PINE &   (1<<PINE2)) ?  FALSE : TRUE)
#define  Is_not_hwb()         ((PINE &   (1<<PINE2)) ?  TRUE : FALSE)
      //! @}


      //! @defgroup EVK_527_df_evk527 ATMEL Hardware data flash configuration
      //! Macros to init the environnement for DF on evk527
      //! @{

// One AT45DB321C
#define DF_4_MB            
#define DF_NB_MEM          1

// PINB0 : SPI Slave Select pin, this pin must be disable (disable input mode) because it is used by joystick
//         Note: the joystick can work fine in Output mode, because the pull-up resistor is activated and the level pin can be read.
// PINB1 : the SCK signal direction is OUTPUT
// PINB2 : the SI signal direction is OUTPUT
// PINB3 : the SO signal direction is INPUT
// PINE6 : the CS signal direction is OUTPUT
#define  Df_init_spi()     (DDRB  &= ~((1<<PINB3)), \
                            DDRB  |=  ((1<<PINB2)|(1<<PINB1)|(1<<PINB0)), \
                            PORTB |=  ((1<<PINB3)|(1<<PINB2)|(1<<PINB1)|(1<<PINB0)), \
                            DDRE  |=  ((1<<PINE6)), \
                            PORTE |=  ((1<<PINE6)) )
#define  Df_select_0()     (PORTE &= ~(1<<PINE6))
#define  Df_desel_all()    (PORTE |=  (1<<PINE6))
      //! @}

   
      //! @defgroup EVK_527_mmc_sd Hardware mmc sd interface configuration
      //! Macros to init the environnement for mmc /sd on evk527
      //! @{   
// PINB0 : SPI Slave Select pin, this pin must be disable (disable input mode) because it is used by joystick
//         Note: the joystick can work fine in Output mode, because the pull-up resistor is activated and the level pin can be read.
//         This one is used to the MMC CS signal
// PINB1 : the SCK signal direction is OUTPUT
// PINB2 : the SI signal direction is OUTPUT
// PINB3 : the SO signal direction is INPUT
#define   Mmc_init_spi()   (DDRB  &= ~((1<<PINB3)), \
                            DDRB  |=  ((1<<PINB2)|(1<<PINB1)|(1<<PINB0)), \
                            PORTB |=  ((1<<PINB3)|(1<<PINB2)|(1<<PINB1)|(1<<PINB0)) )
#define   Mmc_sd_select()   (PORTB &= ~(1<<PINB0))
#define   Mmc_sd_unselect() (PORTB |=  (1<<PINB0))
      //! @}

//! @}

#define  Enable_DAC_pwm()      Timerhs_enable_pwm_d()
#define  Enable_buck_pwm()     Timerhs_enable_pwm_a()  
        
#define  Set_DAC_pwm(v)        Timerhs_set_compare_d(v)
#define  Set_buck_pwm(v)       Timerhs_set_compare_a(v)

//! EVK 527 ADC Channel Definition
#define  ADC_MIC_CH            0x21 // Enable Single Ended Input on ADC9 pin
// temperature sensor removed on EVK527 Rev2.0.0
#define  ADC_TEMP_CH           0x00 // Enable Single Ended Input on ADC0

#define  Dac_micro_init()      (DDRD &= ~(1<<PIND6))

#ifdef USE_ADC       //!< this define is set in config.h file

//! Get_adc_mic_val.
//!
//! This function performs an ADC conversion from the evk527 MIC channel
//! an returns the 10 bits result in an U16 value.
//!
//! @warning USE_ADC should be defined in config.h
//!
//! @param none
//!
//! @return U16 microphone sample value.
//!
   U16 Get_adc_mic_val(void);

//! Get_adc_temp_val.
//!
//! This function performs an ADC conversion from the evk527 TEMP channel
//! an returns the 10 bits result in an U16 value.
//!
//! @warning USE_ADC should be defined in config.h
//!
//! @param none
//!
//! @return U16 analog sensor temperature value.
//!
   U16 Get_adc_temp_val(void);

//! Read_temperature.
//!
//! This function performs an ADC conversion from the evk527 POT channel
//! an returns the 10 bits result of the temperature (in C) in an S16 value.
//!
//! @warning USE_ADC should be defined in config.h
//!
//! @param none
//!
//! @return S16 temperature value in C.
//!
   S16  Read_temperature(void);

#endif   // USE_ADC

#endif   // TARGET_BOARD==EVK527

#endif   // EVK_527_H
