/**
 * \file
 *
 * \brief USB protocol definitions.
 *
 * This file contains definitions and data structures taken from the
 * USB 2.0 specification.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a USBB module can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#ifndef USB_PROTOCOL_H_INCLUDED
#define USB_PROTOCOL_H_INCLUDED

#include <types.h>

/**
 * USB device state. See the device state diagram in the USB 2.0
 * specification for more information.
 */
enum usb_device_state {
	USB_STATE_ATTACHED = 0,
	USB_STATE_POWERED,
	USB_STATE_DEFAULT,
	USB_STATE_ADDRESS,
	USB_STATE_CONFIGURED,
	USB_STATE_SUSPENDED = 0x80,
	USB_STATE_POWERED_SUSPENDED = USB_STATE_POWERED | USB_STATE_SUSPENDED,
	USB_STATE_DEFAULT_SUSPENDED = USB_STATE_DEFAULT | USB_STATE_SUSPENDED,
	USB_STATE_ADDRESS_SUSPENDED = USB_STATE_ADDRESS | USB_STATE_SUSPENDED,
	USB_STATE_CONFIGURED_SUSPENDED = (USB_STATE_CONFIGURED
					  | USB_STATE_SUSPENDED),
};

/**
 * USB device speed. #USB_SPEED_UNKNOWN indicates that the device has
 * not yet seen a reset (i.e. it is in the POWERED or the ATTACHED
 * state, possibly suspended.)
 */
enum usb_device_speed {
	USB_SPEED_UNKNOWN = 0,
	USB_SPEED_LOW,
	USB_SPEED_FULL,
	USB_SPEED_HIGH,
};

/**
 * Possible USB request types (bmRequestType).
 */
enum usb_reqtype {
	USB_REQTYPE_STANDARD	= 0,
	USB_REQTYPE_CLASS	= 1,
	USB_REQTYPE_VENDOR	= 2,
};

/**
 * Possible USB recipient codes (bmRequestType).
 */
enum usb_recipient {
	USB_RECIP_DEVICE	= 0,
	USB_RECIP_INTERFACE	= 1,
	USB_RECIP_ENDPOINT	= 2,
	USB_RECIP_OTHER		= 3,
};

/**
 * Standard USB requests (bRequest).
 */
enum usb_reqid {
	USB_REQ_GET_STATUS		=  0,
	USB_REQ_CLEAR_FEATURE		=  1,
	USB_REQ_SET_FEATURE		=  3,
	USB_REQ_SET_ADDRESS		=  5,
	USB_REQ_GET_DESCRIPTOR		=  6,
	USB_REQ_SET_DESCRIPTOR		=  7,
	USB_REQ_GET_CONFIGURATION	=  8,
	USB_REQ_SET_CONFIGURATION	=  9,
	USB_REQ_GET_INTERFACE		= 10,
	USB_REQ_SET_INTERFACE		= 11,
	USB_REQ_SYNCH_FRAME		= 12,
};

/**
 * Standard USB device feature/status flags.
 */
enum usb_device_feature {
	USB_DEV_FEATURE_REMOTE_WAKEUP	= 1,
	USB_DEV_FEATURE_TEST_MODE	= 2,
};

/**
 * Standard USB endpoint feature/status flags.
 */
enum usb_endpoint_feature {
	USB_EP_FEATURE_HALT		= 0,
};

/**
 * Standard USB Test Mode Selectors
 */
enum usb_test_mode_selector {
	USB_TEST_J			= 0x01,
	USB_TEST_K			= 0x02,
	USB_TEST_SE0_NAK		= 0x03,
	USB_TEST_PACKET			= 0x04,
	USB_TEST_FORCE_ENABLE		= 0x05,
};

/**
 * Standard USB descriptor types.
 */
enum usb_descriptor_type {
	USB_DT_DEVICE			= 1,
	USB_DT_CONFIGURATION		= 2,
	USB_DT_STRING			= 3,
	USB_DT_INTERFACE		= 4,
	USB_DT_ENDPOINT			= 5,
	USB_DT_DEVICE_QUALIFIER		= 6,
	USB_DT_OTHER_SPEED_CONFIGURATION = 7,
	USB_DT_INTERFACE_POWER		= 8,
};

enum usb_ep_xfer_type {
	USB_EP_XFER_CONTROL		= 0x00,
	USB_EP_XFER_ISOCHRONOUS		= 0x01,
	USB_EP_XFER_BULK		= 0x02,
	USB_EP_XFER_INTERRUPT		= 0x03,
};

#define USB_EP_ADDR_MASK		0x0f
#define USB_DIR_IN			0x80
#define USB_DIR_OUT			0x00

/**
 * A USB Device request. The data payload of SETUP packets always
 * follows this structure.
 */
struct usb_setup_req {
	uint8_t bmRequestType;
	uint8_t bRequest;
	le16_t wValue;
	le16_t wIndex;
	le16_t wLength;
} __packed;

#define usb_req_is_in(req)	(!!((req)->bmRequestType & USB_DIR_IN))
#define usb_req_is_out(req)	(!((req)->bmRequestType & USB_DIR_IN))
#define usb_req_type(req)	(((req)->bmRequestType >> 5) & 3)
#define usb_req_recipient(req)	((req)->bmRequestType & 0x1f)

#define USB_MAX_DESC_SIZE	255

/**
 * Standard USB device descriptor.
 */
struct usb_device_descriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	le16_t bcdUSB;
	uint8_t bDeviceClass;
	uint8_t bDeviceSubClass;
	uint8_t bDeviceProtocol;
	uint8_t bMaxPacketSize0;
	le16_t idVendor;
	le16_t idProduct;
	le16_t bcdDevice;
	uint8_t iManufacturer;
	uint8_t iProduct;
	uint8_t iSerialNumber;
	uint8_t bNumConfigurations;
} __packed;

/**
 * Standard USB device qualifier descriptor. This contains information
 * about the device when running at the "other" speed (i.e. if the
 * device is currently operating at high speed, this descriptor can be
 * used to determine what would change if the device was operating at
 * full speed.)
 */
struct usb_device_qualifier_descriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	le16_t bcdUSB;
	uint8_t bDeviceClass;
	uint8_t bDeviceSubClass;
	uint8_t bDeviceProtocol;
	uint8_t bMaxPacketSize0;
	uint8_t bNumConfigurations;
	uint8_t bReserved;
} __packed;

/**
 * Standard USB configuration descriptor.
 */
struct usb_configuration_descriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	le16_t wTotalLength;
	uint8_t bNumInterfaces;
	uint8_t bConfigurationValue;
	uint8_t iConfiguration;
	uint8_t bmAttributes;
	uint8_t bMaxPower;
} __packed;

#define USB_CONFIG_ATTR_MUST_SET	(1U << 7) //!< Must always be set
#define USB_CONFIG_ATTR_SELF_POWERED	(1U << 6) //!< Self-powered
#define USB_CONFIG_ATTR_REMOTE_WAKEUP	(1U << 5) //!< Supports remote wakeup
#define USB_CONFIG_MAX_POWER(ma)	(((ma) + 1) / 2) //!< Max power in mA

/**
 * Standard USB interface descriptor.
 */
struct usb_interface_descriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	uint8_t bInterfaceNumber;
	uint8_t bAlternateSetting;
	uint8_t bNumEndpoints;
	uint8_t bInterfaceClass;
	uint8_t bInterfaceSubClass;
	uint8_t bInterfaceProtocol;
	uint8_t iInterface;
} __packed;

/**
 * Standard USB endpoint descriptor.
 */
struct usb_endpoint_descriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	uint8_t bEndpointAddress;
	uint8_t bmAttributes;
	le16_t wMaxPacketSize;
	uint8_t bInterval;
} __packed;

#define usb_ep_index(desc)	((desc)->bEndpointAddress & USB_EP_ADDR_MASK)
#define usb_ep_is_in(desc)	(!!((desc)->bEndpointAddress & USB_DIR_IN))
#define usb_ep_is_out(desc)	(!((desc)->bEndpointAddress & USB_DIR_IN))
#define usb_ep_xfer(desc)	((desc)->bmAttributes & 0x03)

struct usb_string_descriptor {
	uint8_t	bLength;
	uint8_t	bDescriptorType;
	le16_t	bString[0];
};

#endif /* USB_PROTOCOL_H_INCLUDED */
