/**
 * \file
 *
 * \brief Master SPI driver interface
 *
 * This is the interface to master SPI driver which supports asynchronous
 * operation.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef SPI_H_INCLUDED
#define SPI_H_INCLUDED

#include <stdint.h>
#include <stdbool.h>
#include <compiler.h>
#include <assert.h>
#include <gpio/portmux_gpio.h>
#include <buffer.h>
#include <dmac/dma_controller.h>

#ifdef CONFIG_SPI_USE_DMA
# include "dma.h"
#else
/*
 * We will use the workqueue to achieve async behaviour
 * without drowning in recursive function/callback calls
 */
# include <workqueue.h>
#endif

/** Configuration of a SPI device */
typedef uint32_t spi_config_t;

/**
 * Data associated with a SPI module
 */
struct spi_master
{
	void *port; /*!< Register base */
	struct spi_request *first_request;
	struct spi_request **last_next;
#ifdef CONFIG_SPI_USE_DMA
	struct dmac_channel *dma_tx_channel;
	struct dmac_channel *dma_rx_channel;
	struct dmac_request dma_tx_request;
	struct dmac_request dma_rx_request;
#else
	struct workqueue_item workqueue_item;
#endif
};

/**
 * Data associated with a SPI slave device
 */
struct spi_device
{
	struct spi_master *master;
	spi_config_t config;
	gpio_pin_t cs_pin;
};

enum local_buffer {
	LOCAL_TXBUF,
	LOCAL_RXBUF,
	NONE,
};

/** SPI request flag: Deselect device after transfer */
#define SPI_REQUEST_STOP	(1 << 0)

/**
 * Data associated with a SPI request.
 */
struct spi_request
{
	struct spi_device *device;
	struct buffer *txbuf;
	struct buffer *rxbuf;
	size_t bytes_transferred;
	int status;
	enum local_buffer local_buffer;
	unsigned long flags;
	void (*request_done)(struct spi_request *request, void *context);
	void *context;
	struct spi_request *next;
};

#define SPI_CPHA	(1 << 0)
#define SPI_CPOL	(1 << 1)
#define SPI_MODE_0	0
#define SPI_MODE_1	(SPI_CPHA)
#define SPI_MODE_2	(SPI_CPOL)
#define SPI_MODE_3	(SPI_CPOL | SPI_CPHA)

#ifdef __cplusplus
extern "C" {
#endif

void spi_setup_master(struct spi_master *master, unsigned int id);

/**
 * \brief Configure the portmux to enable a SPI chipselect pin.
 *
 * \param pin The SPI chipselect pin to enable.
 */
static inline void portmux_enable_spi_cs(gpio_pin_t pin)
{
	portmux_select_gpio_pin(pin, PORTMUX_DIR_OUTPUT | PORTMUX_INIT_HIGH);
}

void spi_setup_device(
		struct spi_master *master,
		struct spi_device *device,
		unsigned long flags,
		uint32_t baudrate,
		gpio_pin_t cs_pin);

void spi_submit_request(
		struct spi_device *device,
		struct spi_request *request);

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif  //SPI_H_INCLUDED

