/**
 * \file
 *
 * \brief Linker script definitions
 *
 * This file contains common linker script elements which are the same
 * regardless of how the image ends up being booted.
 *
 * - Compiler:           GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#ifndef LINKERSCRIPT_H_INCLUDED
#define LINKERSCRIPT_H_INCLUDED

/**
 * \brief NanoTrace buffer section.
 *
 * When CONFIG_NANOTRACE_SIZE is defined, this will define a .nanotrace
 * section which will contain a buffer to store NanoTrace data when
 * enabled by an external debugger.
 *
 * When enabled, the start-xxx.S file will define a suitably-sized array
 * called NANOTRACE to be placed in this section.
 */
#define LDS_NANOTRACE_SECTION				\
	.nanotrace (NOLOAD) : {				\
		KEEP(*(.nanotrace));			\
		. = ALIGN(8);				\
	}

/**
 * \brief The stack section.
 *
 * This macro must be used at the end of the linker script to reserve
 * room for the stack.
 */
#define LDS_STACK_SECTION				\
	.stack (NOLOAD) : {				\
		. = . + CONFIG_STACK_SIZE;		\
		. = ALIGN(8);				\
		_end = .;				\
	}

/**
 * \brief Interrupt handler section.
 *
 * This section contains the low-level interrupt entry points.
 */
#define TEXT_IRQ					\
	*(.libavr32.text.irq.*)
/**
 * \brief Interrupt handler data section.
 *
 * This section contains the data pointers associated with each
 * interrupt.
 */
#define BSS_IRQ						\
	*(.libavr32.bss.irq.*)

#ifndef __ASSEMBLY__

/**
 * \brief Exception Vector Base Address
 *
 * This symbol marks the beginning of the exception handler table.
 */
extern void _evba(void);
/**
 * \brief The end of the static data.
 *
 * This symbol marks the end of the statically allocated data,
 * including the stack.
 */
extern unsigned char _end[];

#endif

#endif /* LINKERSCRIPT_H_INCLUDED */
