/**
 * \file
 *
 * \brief Implementation of atomic operations
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC v4.2+ for AVR32
 * - Supported devices:  All AVR32 devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#ifndef ATOMIC_H_INCLUDED
#define ATOMIC_H_INCLUDED

#include <compiler.h>
#include <stdint.h>

typedef struct {
	uint32_t	value;
} atomic_t;

static inline uint32_t atomic_read(atomic_t *ptr)
{
	barrier();
	return ptr->value;
}

static inline void atomic_write(atomic_t *ptr, uint32_t value)
{
	ptr->value = value;
	barrier();
}

/**
 * \brief Atomically add \a value to the value stored at \a ptr
 * \return The result of the addition
 */
static inline atomic_t atomic_add(atomic_t *ptr, uint32_t value)
{
	atomic_t result;

	result.value = __sync_add_and_fetch(&ptr->value, value);
	return result;
}

/**
 * \brief Atomically subtract \a value from the value stored at \a ptr
 * \return The result of the subtraction
 */
static inline atomic_t atomic_sub(atomic_t *ptr, uint32_t value)
{
	atomic_t result;

	result.value = __sync_sub_and_fetch(&ptr->value, value);
	return result;
}

#define atomic_inc(ptr)		atomic_add(ptr, 1)
#define atomic_dec(ptr)		atomic_sub(ptr, 1)

#endif /* ATOMIC_H_INCLUDED */
