/**
 * \file
 *
 * Copyright (C) 2009 Atmel Corporation. All rights reserved.
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/*!
 * \file
 *
 * \brief USBB register definitions
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a USBB module can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Copyright (C) 2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef USBB_REGS_H_INCLUDED
#define USBB_REGS_H_INCLUDED

#include <stdint.h>

/** \name USBB DMA registers */
//@{
#define USBB_DMA_NEXTDESC	0x0000	//!< Next descriptor address
#define USBB_DMA_ADDR		0x0004	//!< DMA Buffer address
#define USBB_DMA_CONTROL	0x0008	//!< DMA Control Register
# define USBB_DMA_CH_EN			(1 <<  0)	//!< Channel enabled
# define USBB_DMA_LD_NXT_CH_DESC_EN	(1 <<  1)	//!< Load next desc
# define USBB_DMA_BUFF_CLOSE_IN_EN	(1 <<  2)	//!< RX done on short
# define USBB_DMA_DMAEND_EN		(1 <<  3)	//!< Last buffer
# define USBB_DMA_EOT			(1 <<  4)	//!< End of transfer
# define USBB_DMA_EOBUFF		(1 <<  5)	//!< End of buffer
# define USBB_DMA_DESC_LD		(1 <<  6)	//!< Descriptor loaded
# define USBB_DMA_BURST_LOCK_EN		(1 <<  7)	//!< Burst lock enable
# define USBB_DMA_BYTE_LEN(x)		((x) << 16)	//!< Bytes remaining
# define USBB_DMA_GET_BYTE_LEN(reg)	(((reg) >> 16) & 0xffff)

#define USBB_DMA_STATUS		0x000c	//!< DMA Status Register
# define USBB_DMA_CH_ACTIVE		(1 <<  1)	//!< Channel active
//@}

/** \name USBB Device Registers */
//@{
#define USBB_UDC_UDCON		0x0000	//!< Control Register
# define USBB_UDCON_UADD(addr)		((addr) <<  0)	//!< Device address
# define USBB_UDCON_UADD_MASK		(0x7f <<  0)
# define USBB_UDCON_ADDEN		(1 <<  7)	//!< Address enable
# define USBB_UDCON_DETACH		(1 <<  8)	//!< Detach from bus
# define USBB_UDCON_RMWKUP		(1 <<  9)	//!< Send remote wakeup
# define USBB_UDCON_SPDCONF_MASK	(3 << 10)	//!< Force bus speed
#  define USBB_UDCON_SPDCONF_NORMAL	(0 << 10)	//!< Dual-speed
#  define USBB_UDCON_SPDCONF_FULL	(3 << 10)	//!< Full-speed
# define USBB_UDCON_LS			(1 << 12)	//!< Low-speed
# define USBB_UDCON_TSTJ		(1 << 13)	//!< Test_J mode
# define USBB_UDCON_TSTK		(1 << 14)	//!< Test_K mode
# define USBB_UDCON_TSTPCKT		(1 << 15)	//!< Test_Packet mode
# define USBB_UDCON_OPMODE2		(1 << 16)	//!< Disable
							//!< bit-stuffing
							//!< and NRZI encoding
#define USBB_UDC_UDINT		0x0004	//!< Interrupt Register
#define USBB_UDC_UDINTCLR	0x0008	//!< Interrupt Clear
#define USBB_UDC_UDINTSET	0x000c	//!< Interrupt Set
#define USBB_UDC_UDINTE		0x0010	//!< Interrupt Enable
#define USBB_UDC_UDINTECLR	0x0014	//!< Interrupt Enable Clear
#define USBB_UDC_UDINTESET	0x0018	//!< Interrupt Enable Set
# define USBB_UDINT_SUSP		(1 <<  0)	//!< Device suspended
# define USBB_UDINT_MSOF		(1 <<  1)	//!< Start of uFrame
# define USBB_UDINT_SOF			(1 <<  2)	//!< Start of frame
# define USBB_UDINT_EORST		(1 <<  3)	//!< End of reset
# define USBB_UDINT_WAKEUP		(1 <<  4)	//!< Wake up device
# define USBB_UDINT_EORSM		(1 <<  5)	//!< End of resume
# define USBB_UDINT_UPRSM		(1 <<  6)	//!< Upstream resume
# define USBB_UDINT_EP(x)		(1 << ((x) + 12)) //!< EP interrupt
# define USBB_UDINT_DMA(x)		(1 << ((x) + 24)) //!< DMA interrupt

#define USBB_UDC_UERST		0x001c	//!< Endpoint Enable / Reset
# define USBB_UERST_EPEN(x)		(1 << ((x) +  0)) //!< Endpoint enable
# define USBB_UERST_EPRST(x)		(1 << ((x) + 16)) //!< Endpoint reset

#define USBB_UDC_UDFNUM		0x0020	//!< Frame Number
# define USBB_UDFNUM_GET_MFNUM(reg)	((reg) & 7)	//!< uFrame number
# define USBB_UDFNUM_GET_FNUM(reg)	(((reg) >> 3) & 0x7ff) //!< frame num
# define USBB_UDFNUM_FNCERR		(1 << 15)	//!< CRC error

#define USBB_UDC_UDTST1		0x0024	//!< Test Register 1
#define USBB_UDC_UDTST2		0x0028	//!< Test Register 2

/** \name USBB Endpoint Registers */
//@{
#define USBB_UDC_UECFG(x)	(0x100 + (x) * 4)	//!< Endpoint Config
# define USBB_UECFG_ALLOC		(1 <<  1)	//!< Allocate ep mem
# define USBB_UECFG_EPBK(x)		((x) << 2)	//!< Number of banks
# define USBB_UECFG_EPSIZE(x)		((x) << 4)	//!< Endpoint size
# define USBB_UECFG_EPDIR_OUT		(0 <<  8)	//!< OUT or control
# define USBB_UECFG_EPDIR_IN		(1 <<  8)	//!< IN direction
# define USBB_UECFG_EPAUTOSW		(1 <<  9)	//!< Automatic switch
# define USBB_UECFG_EPTYPE(x)		((x) << 11)	//!< Endpoint type
# define USBB_UECFG_NBTRANS(x)		((x) << 13)	//!< Trans/uframe

#define USBB_UDC_UESTA(x)	(0x130 + (x) * 4)	//!< EPx Status
#define USBB_UDC_UESTACLR(x)	(0x160 + (x) * 4)	//!< EPx Status Clear
#define USBB_UDC_UESTASET(x)	(0x190 + (x) * 4)	//!< EPx Status Set
# define USBB_UESTA_GET_DTSEQ(x)	(((x) >> 8) & 3) //!< Current data PID
# define USBB_UESTA_GET_NBUSYBK(x)	(((x) >> 12) & 3) //!< # busy banks
# define USBB_UESTA_GET_CURRBK(x)	(((x) >> 14) & 3) //!< Current bank
# define USBB_UESTA_RWALL		(1 << 16)	//!< Read/write allowed
# define USBB_UESTA_CTRLDIR_IN		(1 << 17)	//!< Control direction
# define USBB_UESTA_CFGOK		(1 << 18)	//!< Ep config OK
# define USBB_UESTA_GET_BYCT(x)		(((x) >> 20) & 0x7ff) //!< Byte count

#define USBB_UDC_UECON(x)	(0x1c0 + (x) * 4)	//!< EPx Ctrl
#define USBB_UDC_UECONSET(x)	(0x1f0 + (x) * 4)	//!< EPx Ctrl Set
#define USBB_UDC_UECONCLR(x)	(0x220 + (x) * 4)	//!< EPx Ctrl Clear
# define USBB_UECON_MDATAE		(1 <<  8)	//!< MDATA int enable
# define USBB_UECON_DATAXE		(1 <<  9)	//!< DATAX int enable
# define USBB_UECON_NBUSYBKE		(1 << 12)	//!< Idle int enable
# define USBB_UECON_KILLBK		(1 << 13)	//!< Free last IN bank
# define USBB_UECON_FIFOCON		(1 << 14)	//!< FIFO control
# define USBB_UECON_EPDISHDMA		(1 << 16)	//!< Int stops DMA
# define USBB_UECON_NYETDIS		(1 << 17)	//!< Disable NYET
# define USBB_UECON_RSTDT		(1 << 18)	//!< Reset Data Toggle
# define USBB_UECON_STALLRQ		(1 << 19)	//!< Send STALL

//! \name Endpoint Interrupt bits
//@{
# define USBB_EP_TXINI			(1 <<  0)	//!< TX IN ready
# define USBB_EP_RXOUTI			(1 <<  1)	//!< RX OUT ready
# define USBB_EP_RXSTPI			(1 <<  2)	//!< RX SETUP ready
# define USBB_EP_UNDERFI		(1 <<  2)	//!< Iso flow error
# define USBB_EP_NAKOUTI		(1 <<  3)	//!< OUT NAK sent
# define USBB_EP_HBISOINERRI		(1 <<  3)	//!< HB iso underflow
# define USBB_EP_NAKINI			(1 <<  4)	//!< IN NAK sent
# define USBB_EP_HBISOFLUSHI		(1 <<  4)	//!< HB iso flushed
# define USBB_EP_OVERFI			(1 <<  5)	//!< FIFO overflow
# define USBB_EP_STALLEDI		(1 <<  6)	//!< STALL sent
# define USBB_EP_CRCERRI		(1 <<  6)	//!< Iso CRC error
# define USBB_EP_SHORTI			(1 <<  7)	//!< Short packet
# define USBB_EP_ERRORTRANSI		(1 << 10)	//!< Iso PID error
//@}
//@}
//@}

//! \name USBB Host Registers
//@{
#define USBB_HOST_UHCON		0x0400	//!< Host Control
#define USBB_HOST_UHINT		0x0404	//!< Interrupt
#define USBB_HOST_UHINTCLR	0x0408	//!< Interrupt Clear
#define USBB_HOST_UHINTSET	0x040c	//!< Interrupt Set
#define USBB_HOST_UHINTE	0x0410	//!< Interrupt Enable
#define USBB_HOST_UHINTECLR	0x0414	//!< Interrupt Enable Clear
#define USBB_HOST_UHINTESET	0x0418	//!< Interrupt Enable Set
#define USBB_HOST_UPRST		0x041c	//!< Pipe Enable/Reset
#define USBB_HOST_UHFNUM	0x0420	//!< Frame Number
#define USBB_HOST_UHADDR1	0x0424	//!< Host Address 1
#define USBB_HOST_UHADDR2	0x0428	//!< Host Address 2

//! \name USBB Pipe Registers
//@{
#define USBB_HOST_UPCFG(x)	(0x500 + 4 * (x))	//!< Pipe Config
#define USBB_HOST_UPSTA(x)	(0x530 + 4 * (x))	//!< Pipe Status
#define USBB_HOST_UPSTACLR(x)	(0x560 + 4 * (x))	//!< Pipe Status Clear
#define USBB_HOST_UPSTASET(x)	(0x590 + 4 * (x))	//!< Pipe Status Set
#define USBB_HOST_UPCON(x)	(0x5c0 + 4 * (x))	//!< Pipe Ctrl
#define USBB_HOST_UPCONSET(x)	(0x5f0 + 4 * (x))	//!< Pipe Ctrl Set
#define USBB_HOST_UPCONCLR(x)	(0x620 + 4 * (x))	//!< Pipe Ctrl Clear
#define USBB_HOST_UPINRQ(x)	(0x650 + 4 * (x))	//!< Pipe IN Request
#define USBB_HOST_UPERR(x)	(0x680 + 4 * (x))	//!< Pipe Error
//@}
//@}

//! \name USBB Core Registers
//@{
#define USBB_CORE_USBCON	0x0800	//!< USB Control
# define USBB_USBCON_VBUSHWC		(1 <<  8)	//!< Vbus HW control
# define USBB_USBCON_SRPSEL		(1 <<  9)	//!< SRP method
# define USBB_USBCON_SRPREQ		(1 << 10)	//!< SRP request
# define USBB_USBCON_HNPREQ		(1 << 11)	//!< HNP request
# define USBB_USBCON_OTGPADE		(1 << 12)	//!< OTG pad enable
# define USBB_USBCON_VBUSPO		(1 << 13)	//!< Vbus polarity
# define USBB_USBCON_FRZCLK		(1 << 14)	//!< Freeze clock
# define USBB_USBCON_USBE		(1 << 15)	//!< Enable controller
# define USBB_USBCON_TIMVALUE(x)	((x) << 16)	//!< Timer value
# define USBB_USBCON_TIMPAGE(x)		((x) << 20)	//!< Timer page
# define USBB_USBCON_UNLOCK		(1 << 22)	//!< Timer unlock
# define USBB_USBCON_UIDE		(1 << 24)	//!< ID pin enable
# define USBB_USBCON_UIMOD_HOST		(0 << 25)	//!< Host mode
# define USBB_USBCON_UIMOD_DEVICE	(1 << 25)	//!< Device mode

#define USBB_CORE_USBSTA	0x0804	//!< USB Status
#define USBB_CORE_USBSTACLR	0x0808	//!< USB Status Clear
#define USBB_CORE_USBSTASET	0x080c	//!< USB Status Set
# define USBB_USBSTA_VBUSRQ		(1 <<  9)	//!< Vbus request
# define USBB_USBSTA_ID			(1 << 10)	//!< ID pin state
# define USBB_USBSTA_VBUS		(1 << 11)	//!< Vbus available
# define USBB_USBSTA_SPEED_MASK		(3 << 12)	//!< Bus connection
#  define USBB_USBSTA_SPEED_FULL	(0 << 12)	//!< Full-speed
#  define USBB_USBSTA_SPEED_HIGH	(1 << 12)	//!< High-speed
#  define USBB_USBSTA_SPEED_LOW		(2 << 12)	//!< Low-speed
# define USBB_USBSTA_CLKUSABLE		(1 << 14)	//!< UTMI clock OK

//! \name USBB Core Interrupt Bits
//@{
# define USBB_IDTI			(1 <<  0)	//!< ID transition
# define USBB_VBUSTI			(1 <<  1)	//!< Vbus transition
# define USBB_SRPI			(1 <<  2)	//!< SRP detected
# define USBB_VBERRI			(1 <<  3)	//!< Vbus error
# define USBB_BCERRI			(1 <<  4)	//!< B conn error
# define USBB_ROLEEXI			(1 <<  5)	//!< Roles switched
# define USBB_HNPERRI			(1 <<  6)	//!< HNP error
# define USBB_STOI			(1 <<  7)	//!< Suspend time-out

#define USBB_CORE_USBFSM	0x082c	//! USB OTG State
//@}
//@}

struct usbb_dma_desc {
	uint32_t	next;
	uint32_t	buf_addr;
	uint32_t	control;
};

/* Register access macros */
#define usbb_read_reg(reg)						\
	mmio_read32((void *)(USBB_REGS_BASE + USBB_CORE_##reg))
#define usbb_write_reg(reg, value)					\
	mmio_write32((void *)(USBB_REGS_BASE + USBB_CORE_##reg), value)

#define usbb_udc_read_reg(reg)						\
	mmio_read32((void *)(USBB_REGS_BASE + USBB_UDC_##reg))
#define usbb_udc_write_reg(reg, value)					\
	mmio_write32((void *)(USBB_REGS_BASE + USBB_UDC_##reg), value)

#define usbb_udc_dma_read_reg(ep, reg)					\
	mmio_read32((void *)(USBB_REGS_BASE + 0x300 + (ep) * 0x10	\
				+ USBB_DMA_##reg))
#define usbb_udc_dma_write_reg(ep, reg, value)				\
	mmio_write32((void *)(USBB_REGS_BASE + 0x300 + (ep) * 0x10	\
				+ USBB_DMA_##reg), value)

/* Get the FIFO address of an endpoint */
#define usbb_ep_fifo(ep)	((void *)(USBB_FIFO_BASE + ep * 0x10000))

#endif /* USBB_REGS_H_INCLUDED */
