/**
 * \file
 *
 * Copyright (C) 2009 Atmel Corporation. All rights reserved.
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/*!
 * \file
 *
 * \brief USBB driver: Internal functions and definitions
 *
 * This header is for internal use only. There is lots of ugly
 * preprocessor stuff in here so that the rest of the code can be
 * reasonably clean and efficient.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a USBB module can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Copyright (C) 2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef USBB_INTERNAL_H_INCLUDED
#define USBB_INTERNAL_H_INCLUDED

#include <bitops.h>
#include <dmapool.h>
#include <slist.h>
#include <stdint.h>
#include <usb/request.h>
#include <usb/udc.h>

#include "usbb_regs.h"

/**
 * \internal
 * \brief USBB DMA descriptor.
 */
struct usbb_sw_dma_desc {
	/**
	 * \brief Hardware DMA descriptor.
	 */
	struct usbb_dma_desc	hw;
	/**
	 * \brief The physical address of this descriptor.
	 * \todo not necessary on all CPUs
	 */
	phys_addr_t			phys;
};

static inline phys_addr_t usbb_dma_desc_phys(struct usbb_sw_dma_desc *desc)
{
	return desc->phys;
};

static inline void usbb_dma_desc_set_phys(struct usbb_sw_dma_desc *desc,
		phys_addr_t phys)
{
	desc->phys = phys;
}

/* USB Device Controller part */

/**
 * Control endpoint state.
 */
enum ep0_state {
	EP0_STATE_SETUP = 0,	/**< Waiting for SETUP */
	EP0_STATE_DATA_IN,	/**< IN data being sent */
	EP0_STATE_DATA_ZLP,	/**< Zero-Length Packet being sent */
	EP0_STATE_DATA_OUT,	/**< Waiting for OUT data */
	EP0_STATE_STATUS_IN,	/**< IN status sent */
	EP0_STATE_STATUS_OUT,	/**< Waiting for OUT status */
	EP0_STATE_TEST_J,	/**< Test_J mode after status */
	EP0_STATE_TEST_K,	/**< Test_K mode after status */
	EP0_STATE_TEST_SE0_NAK,	/**< Test_SE0_NAK mode after status */
	EP0_STATE_TEST_PACKET,	/**< Test_Packet mode after status */
};

enum usbb_ep_flag {
	USBB_EP_ACTIVE_XFER,
	USBB_EP_ENABLED,
	USBB_EP_WEDGE,
};

struct usbb_udc_ep {
	const struct usb_endpoint_descriptor *desc;
	unsigned int		buf_offset;
	uint16_t		maxpacket;
	uint16_t		bytes_written;
	unsigned long		flags;
	struct slist		req_queue;
	struct slist		buf_queue;
};

struct usbb_udc {
	struct udc		udc;
	enum ep0_state		ctrl_state;
	struct usb_setup_req	setup_req;
	struct usbb_controller	*usbb;
	struct usbb_udc_ep	ep[CONFIG_USBB_NR_EP];
};

static inline bool usbb_udc_entering_test_mode(struct usbb_udc *udcb)
{
#ifdef CONFIG_UDC_HIGH_SPEED
	return udcb->ctrl_state >= EP0_STATE_TEST_J
		&& udcb->ctrl_state <= EP0_STATE_TEST_PACKET;
#else
	return false;
#endif
}

static inline bool usbb_udc_is_enabled(struct usbb_udc *udcb)
{
#if defined(CONFIG_USBB_OTG)
	return test_bit(USB_DEV_IS_ENABLED, &udcb->udc.flags);
#elif defined(CONFIG_USBB_UDC)
	return true;
#else
	return false;
#endif
}

static inline void usbb_udc_enable(struct usbb_udc *udcb)
{
	/*
	 * Keep the autoattach flag as the application left it, set the
	 * enabled flag and clear everything else. The USBB core driver
	 * will check the Vbus state right after enabling the
	 * controller.
	 */
	udcb->udc.flags &= (1 << USB_DEV_IS_ENABLED)
		| (1 << USB_DEV_AUTOATTACH);
	udcb->udc.flags |= 1 << USB_DEV_IS_ENABLED;
}

static inline void usbb_udc_disable(struct usbb_udc *udcb)
{
	/*
	 * Clear all flags except the autoattach flag set by the
	 * application
	 */
	udcb->udc.flags &= 1 << USB_DEV_AUTOATTACH;
}

extern void usbb_udc_interrupt(struct usbb_udc *udcb);
extern void usbb_udc_vbus_off(struct usbb_udc *udcb);
extern void usbb_udc_vbus_on(struct usbb_udc *udcb);
extern struct usbb_udc *usbb_udc_init(void);
extern void usbb_udc_shutdown(struct usbb_udc *udcb);

/* USB Host part */

struct usbb_host {

};

static inline bool usbb_host_is_enabled(struct usbb_host *hostb)
{
#if defined(CONFIG_USBB_OTG)
	return test_bit(USB_HOST_IS_ENABLED, &hostb->hcd.flags);
#elif defined(CONFIG_USBB_HOST)
	return true;
#else
	return false;
#endif
}

extern void usbb_host_enable(struct usbb_host *hostb);
extern void usbb_host_disable(struct usbb_host *hostb);
extern struct usbb_host *usbb_host_init(void);
extern void usbb_host_shutdown(struct usbb_host *hostb);

/* USB Master and OTG part */

struct usbb_controller {
#ifdef CONFIG_USBB_UDC
	struct usbb_udc		*udc;
#endif
#ifdef CONFIG_USBB_HOST
	struct usbb_host	*host;
#endif
};

extern struct dma_pool	usbb_desc_pool;

static inline struct usbb_sw_dma_desc *usbb_dma_desc_alloc(phys_addr_t *paddr)
{
	return dma_pool_alloc(&usbb_desc_pool, paddr);
}

static inline void usbb_dma_desc_free(struct usbb_sw_dma_desc *desc)
{
	dma_pool_free(&usbb_desc_pool, desc);
}

#endif /* USBB_INTERNAL_H_INCLUDED */
