/**
 * \file
 *
 * \brief Dummy block device used for testing
 *
 * This block device will silently consume any data written to it and will
 * generate zeroes as data when read.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#include <assert.h>
#include <bitops.h>
#include <debug.h>
#include <malloc.h>
#include <status-codes.h>
#include <string.h>
#include <block/device.h>
#include <block/dummy.h>

static void dummy_do_read(struct block_device *bdev, struct block_request *req)
{
	struct buffer	*buf;
	size_t		bytes_read = 0;

	blk_req_for_each_buffer(req, buf) {
		bytes_read += buf->len;
		memset(buf->addr.ptr, 0, buf->len);
	}

	req->bytes_xfered = bytes_read;
	req->status = 0;

	req->req_done(bdev, req);
	free(req);
}

static void dummy_do_write(struct block_device *bdev, struct block_request *req)
{
	struct buffer	*buf;
	size_t		bytes_written = 0;

	blk_req_for_each_buffer(req, buf)
		bytes_written += buf->len;

	req->bytes_xfered = bytes_written;
	req->status = 0;

	req->req_done(bdev, req);
	free(req);
}

static void dummy_prepare_req(struct block_device *bdev,
		struct block_request *req,
		uint32_t lba, uint32_t nr_blocks,
		enum block_operation operation)
{
	slist_init(&req->buf_list);
	req->status = -STATUS_IN_PROGRESS;
	req->bytes_xfered = 0;
	switch (operation) {
	case BLK_OP_READ:
		req->req_submit = dummy_do_read;
		break;
	case BLK_OP_WRITE:
		req->req_submit = dummy_do_write;
		break;
	default:
		unhandled_case(operation);
	}
}

static struct block_request *dummy_alloc_req(struct block_device *bdev)
{
	struct block_request	*req;

	req = malloc(sizeof(struct block_request));
	if (unlikely(!req))
		return NULL;
	memset(req, 0, sizeof(struct block_request));

	return req;
}

static void dummy_free_req(struct block_device *bdev, struct block_request *req)
{
	free(req);
}

static uint32_t dummy_get_dev_id(struct block_device *bdev)
{
	return 0;
}

/**
 * \brief Initialize a dummy block device
 *
 * \param block_size The size of each block in this device
 * \param nr_blocks The number of blocks available
 * \param writeable true if the device should be writeable, false if
 * it should be read-only
 *
 * \return A new dummy block device, or NULL if insufficient memory is
 * available.
 */
struct block_device *dummy_blkdev_init(uint16_t block_size, uint32_t nr_blocks,
		bool writeable)
{
	struct block_device *bdev;

	bdev = malloc(sizeof(struct block_device));
	if (!bdev)
		return NULL;
	memset(bdev, 0, sizeof(struct block_device));

	bdev->block_size = block_size;
	bdev->nr_blocks = nr_blocks;

	if (writeable)
		set_bit(BDEV_WRITEABLE, &bdev->flags);
	set_bit(BDEV_PRESENT, &bdev->flags);

	bdev->prepare_req = dummy_prepare_req;
	bdev->alloc_req = dummy_alloc_req;
	bdev->free_req = dummy_free_req;
	bdev->get_dev_id = dummy_get_dev_id;

	return bdev;
}
