/**
 * \file
 *
 * \brief USB Documentation Overview
 *
 * This file provides an overview of the USB stack. It does not
 * contain any code by itself.
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * \page USB-overview USB Subsystem Overview
 *
 * The subsystem is a framework for implementing applications where
 * the board will act as a USB device, USB host or both.
 *
 * The following diagram is a high-level overview of the USB subsystem
 * architecture. The left side of the "USB Library" box is the USB
 * Device framework, while the right side is the USB Host framework.
 * An application may choose to use both or just one of the
 * frameworks. The "OTG / transceiver" part is optional, but usually
 * present when using a dual-role controller.
 *
 * \dot
	digraph usbarch {
		node [shape=none];
		node [fontname="FreeSans"];
		usb [label=<
<table width="80%" border="0" cellborder="1" cellspacing="0" cellpadding="8">
  <tr>
    <td>Function<br/>Driver</td>
    <td>...</td>
    <td>Function<br/>Driver</td>
    <td rowspan="3">USB<br/>Core<br/>Library</td>
    <td>Interface<br/>Driver</td>
    <td>...</td>
    <td>Interface<br/>Driver</td>
  </tr>
  <tr>
    <td colspan="3">Function Core</td>
    <td colspan="3">USB Host Core</td>
  </tr>
  <tr>
    <td colspan="3" cellpadding="0">
      <table border="0" cellborder="1" cellspacing="0" cellpadding="8">
        <tr>
          <td>UDC Library</td>
	  <td port="udc_drv">UDC Driver</td>
	</tr>
      </table>
    </td>
    <td colspan="3" port="usb_hcd">USB Host Controller Driver (HCD)</td>
  </tr>
</table>>];
		otg [label="USB OTG / Transceiver Driver", shape="box"];
		usb:usb_hcd -> otg [dir="both"];
		usb:udc_drv -> otg [dir="both"];
	}
 * \enddot
 *
 * For more information, please see one of the following pages:
 * - \ref usb-lib
 * - \ref USB-device-overview
 *
 * \section usb-lib The USB core library
 *
 * The USB core library provides helper functions which can be used by
 * both device- and host-side USB components. To initialize the USB
 * core library, the following function must be called before any
 * drivers are initialized.
 * - usb_init(): \copybrief usb_init()
 *
 * \subsection usb-request USB request management
 *
 * A USB request contains a list of buffers, each representing a
 * contiguous region of memory used for holding data to be sent or
 * received in a USB transfer. See the documentation of struct
 * usb_request for details.
 *
 * Each request may have a callback associated with it, which is
 * called when all the buffers have been transferred or cancelled. The
 * submitter may use usb_request::context to hold arbitrary data
 * associated with the request.
 *
 * Before a USB request can be (re)used, it must be initialized by
 * calling usb_req_init().
 *
 * USB requests can also be allocated dynamically from the USB request
 * pool using the following functions. These functions will
 * automatically initialize the request, so there's no need to call
 * usb_req_init() on a freshly allocated request.
 * - usb_req_alloc(): \copybrief usb_req_alloc()
 * - usb_req_alloc_single_tx(): \copybrief usb_req_alloc_single_tx()
 * - usb_req_alloc_single_rx(): \copybrief usb_req_alloc_single_rx()
 * - usb_req_free(): \copybrief usb_req_free()
 * - usb_req_free_all(): \copybrief usb_req_free_all()
 *
 * Finally, the following function can be used to associate another
 * buffer with a request.
 * - usb_req_add_buffer(): \copybrief usb_req_add_buffer()
 *
 * \page USB-device-overview The USB Device Framework
 *
 * The USB Device framework consists of the following components:
 * - \ref udc-driver (include/usb/udc.h)
 * - \ref udc-lib (drivers/usb/udc_lib.c)
 * - \ref usb-function-core (drivers/usb/function/usb_function_core.c)
 *
 * \section udc-driver USB Device Controller Interface
 *
 * The UDC driver provides a low-level abstraction of the hardware.
 * Most events originate from the hardware in the form of interrupts,
 * which may cause the UDC driver to call into the UDC library or the
 * function driver. These may in turn call back into the driver in
 * order to send an appropriate response to the host.
 *
 * \subsection udc-driver-init Initialization
 *
 * UDC initialization is driver-specific and not covered by this API.
 * Some drivers may provide an initialization function for the
 * controller as a whole, which may in turn initialize the UDC, host
 * and transceiver parts depending on the configuration as well as
 * controller-wide state changes (Vbus detection, ID pin state, etc.)
 *
 * \subsection udc-ep-mgmt Endpoint Management
 *
 * These functions allow drivers to create and destroy endpoints, as
 * well as set, clear and query their "halted" state:
 * - udc_ep_create(): \copybrief udc_ep_create()
 * - udc_ep_destroy(): \copybrief udc_ep_destroy()
 * - udc_ep_set_halt(): \copybrief udc_ep_set_halt()
 * - udc_ep_clear_halt(): \copybrief udc_ep_clear_halt()
 * - udc_ep_is_halted(): \copybrief udc_ep_is_halted()
 *
 * Sometimes, when encountering errors that need some sort of error
 * recovery procedure to be performed by the host, an endpoint must be
 * halted and the host must be prevented from clearing the halted
 * state through the standard ClearFeature(HALT) request. The
 * following functions can be used to "wedge" an endpoint in this
 * manner:
 * - udc_ep_set_wedge(): \copybrief udc_ep_set_wedge()
 * - udc_ep_clear_wedge(): \copybrief udc_ep_clear_wedge()
 * - udc_ep_is_wedged(): \copybrief udc_ep_is_wedged()
 *
 * Finally, the following function will terminate all queued requests
 * and flush the endpoint FIFO. The data toggle sequencing and
 * halted/wedged states will not be affected.
 * - udc_ep_flush(): \copybrief udc_ep_flush()
 *
 * \subsection udc-control Control Transfers
 *
 * These functions are used to send and receive packets over the
 * default control endpoint, ep0. The following functions are
 * asynchronous and will call a callback function after the transfer
 * is done. They must be used for all OUT transfers, and should be
 * used for most IN transfers.
 * - udc_ep0_submit_out_req(): \copybrief udc_ep0_submit_out_req()
 * - udc_ep0_submit_in_req(): \copybrief udc_ep0_submit_in_req()
 *
 * The following functions are synchronous and should be avoided.
 * However, they can sometimes be useful for sending very small
 * packets.
 * - udc_ep0_write_sync(): \copybrief udc_ep0_write_sync()
 * - udc_ep0_send_status(): \copybrief udc_ep0_send_status()
 *
 * The function driver should notify the driver when to expect a
 * status OUT packet from the host by calling the following function:
 * - udc_ep0_expect_status(): \copybrief udc_ep0_expect_status()
 *
 * \subsection udc-transfer Data Transfer
 *
 * These functions are used to do non-control transfers, i.e. bulk,
 * interrupt and isochronous transfers. Note that no synchronous
 * functions are available -- the caller must always provide a
 * callback if needed.
 * - udc_ep_submit_out_req(): \copybrief udc_ep_submit_out_req()
 * - udc_ep_submit_in_req(): \copybrief udc_ep_submit_in_req()
 *
 * \section udc-lib The USB Device Controller Library
 *
 * The UDC Library contains common helper functions for USB Device
 * drivers. Currently, only a single function is available:
 * - udc_lib_process_setup_request(): \copybrief udc_lib_process_setup_request()
 *
 * This function may be called by the UDC driver to handle a SETUP
 * request received on the default control endpoint (ep0). It will
 * either handle the request directly or pass it on to a higher layer.
 *
 * The following function is called by the UDC Library when a SET
 * ADDRESS request is received from the host:
 * - udc_set_address(): \copybrief udc_set_address()
 *
 * \section usb-function-core The USB Function Driver Core
 *
 * The USB Function Driver Core keeps track of configurations,
 * interfaces and settings, and switches between them based on
 * requests received from the host.
 *
 * A USB device can have one or more configurations, which the host
 * can choose between using the standard SET_CONFIGURATION request.
 * The host can also choose not to enable any configuration by
 * selecting the special configuration ID 0. Each configuration can
 * have one or more interfaces, which are all active at the same time.
 * Each interface can have one or more alternate settings which the
 * host can choose between using the standard SET_INTERFACE request.
 *
 * \subsection udc-device-descriptor The Device Descriptor
 *
 * The USB Function Driver Core also provides a Device Descriptor for
 * the device. The contents of this can be specified by defining the
 * following macros in app/config_usb.h. Please see the USB
 * Specification for more information about the meaning of the fields.
 * - \c CONFIG_USB_FUNC_DEVICE_CLASS: bDeviceClass
 * - \c CONFIG_USB_FUNC_DEVICE_SUBCLASS: bDeviceSubClass
 * - \c CONFIG_USB_FUNC_DEVICE_PROTOCOL: bDeviceProtocol
 * - \c CONFIG_USB_FUNC_MAXPACKETSIZE0: bMaxPacketSize0
 * - \c CONFIG_USB_FUNC_VENDOR_ID: idVendor
 * - \c CONFIG_USB_FUNC_PRODUCT_ID: idProduct
 * - \c CONFIG_USB_FUNC_MAJOR_VERSION /
 *   \c CONFIG_USB_FUNC_MINOR_VERSION: bcdDevice
 * - \c CONFIG_USB_FUNC_NR_CONFIGURATIONS: bNumConfigurations
 *
 * \subsection udc-func-config-iface Creating Configurations and Interfaces
 *
 * The following functions are called by the application to create
 * configurations and associate interfaces with them. The interface
 * objects are typically returned by the initialization function of a
 * Function Driver.
 * - usb_func_add_config(): \copybrief usb_func_add_config()
 * - usb_func_add_interface(): \copybrief usb_func_add_interface()
 *
 * usb_func_add_config() returns a configuration object which can be
 * passed as a parameter to usb_func_add_interface(). The application
 * is responsible for creating a configuration descriptor for each
 * configuration, and it must never add more configurations than
 * specified by the CONFIG_USB_FUNC_NR_CONFIGURATIONS macro, or more
 * interfaces than specified by the configuration descriptor's
 * bNumInterfaces field.
 *
 * \subsection udc-func-drv Function Driver Interface
 *
 * A USB Function Driver instance is represented by a struct
 * usb_func_iface object. Each such object contains one or more struct
 * usb_func_iface_setting objects, which in turn is associated with a
 * set of interface operations (struct usb_func_iface_ops). These
 * operations are called by the Function Driver Core in response to
 * certain requests from the host:
 * - usb_func_iface_ops::enable(): Called when the host selects this
 *   interface setting using SET_INTERFACE, and for interface setting
 *   0 when the host selects the configuration to which this interface
 *   belongs using SET_CONFIGURATION.
 * - usb_func_iface_ops::disable(): Called when this interface setting
 *   is enabled, and a SET_INTERFACE request with this interface as
 *   the target or a SET_CONFIGURATION request is sent by the host.
 * - usb_func_iface_ops::setup(): Called when this interface setting
 *   is enabled, and a vendor- or class-specific request with this
 *   interface as the target is sent by the host.
 *
 * When an interface setting has been enabled, the driver may interact
 * directly with the UDC driver, managing endpoints and submitting
 * requests on them. The function driver may only submit control
 * transfers when it is processing a SETUP request directed at its
 * interface.
 *
 * Note that the function driver must never keep any endpoints active
 * after usb_func_iface_ops::disable() has been called. When the host
 * re-selects the currently active interface setting using
 * SET_INTERFACE, the Function Core will disable and re-enable the
 * setting. This ensures that the endpoints associated with the
 * interface are properly reset.
 *
 * \subsection udc-func-interface Functions called by the UDC driver
 *
 * The following functions are called by the USB Device Controller
 * driver or the UDC library in response to USB bus events and control
 * requests from the host:
 * - usb_func_reset(): \copybrief usb_func_reset()
 * - usb_func_get_descriptor(): \copybrief usb_func_get_descriptor()
 * - usb_func_set_configuration(): \copybrief usb_func_set_configuration()
 * - usb_func_set_interface(): \copybrief usb_func_set_interface()
 * - usb_func_get_interface(): \copybrief usb_func_get_interface()
 * - usb_func_process_setup_request(): \copybrief
 * usb_func_process_setup_request()
 */
