/**
 * \file
 *
 * \brief Portmux and GPIO documentation
 *
 * This file provides an overview of the port multiplexer
 * configuration API. It does not contain any code by itself.
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * \page Portmux-overview Port multiplexer configuration and GPIO API overview
 *
 * On AVR chips, most external I/O pins are routed through a port
 * multiplexer. There are currently two kinds of port multiplexer
 * hardware around with different register interfaces:
 *
 * - PIO (AT32AP700x; also used on ARM AT91 chips)
 * - GPIO (all other AVR32 chips)
 *
 * The "PIO" variant supports multiplexing up to two peripherals per
 * pin in addition to GPIO (software control). Each pin has
 * configurable pull-up, glitch filter, interrupt and multi-drive
 * capabilities.
 *
 * The "GPIO" variant supports multiplexing up to four peripherals per
 * pin in addition to GPIO. Each pin has configurable
 * pull-up/pull-down/buskeeper, glitch filter, interrupt, open-drain
 * and schmitt-trigger capabilities, as well as configurable drive
 * strength and slew rate control.
 *
 * Both controllers are configured using the same API, but the
 * functions may accept different values for some parameters depending
 * on the actual portmux implementation, and some parameters may be
 * ignored by one of the implementation (e.g. the "PIO" implementation
 * will ignore the drive strength flags since the hardware doesn't
 * support configurable drive strength.)
 *
 * \section identifying-pins Identifying Pins
 *
 * The portmux configuration functions described below identify the
 * pins to act on based on two parameters: A "port" (i.e. a block of
 * pins that somehow belong together) and a pin mask. Both are defined
 * in an implementation-specific manner.
 *
 * The available ports are defined on the form
 * \code 
 * #define PORTMUX_PORT_A	(something)
 * \endcode
 *
 * where "A" matches the identifier given in the chip's data sheet,
 * and "something" is whatever the portmux implementation needs to
 * identify the port (usually a memory address).
 *
 * The pin mask is a bitmask of type ::pin_mask_t, where each '1' bit
 * indicates a pin to apply the current operation to. The width of the
 * bitmask is implementation specific, but it's usually 32 bits on
 * 32-bit architectures.
 *
 * \section selecting-functions Selecting Functions
 *
 * Each pin can either be assigned to one of a predefined set of
 * on-chip peripherals, or it can be set up to be controlled by
 * software. For the former case, the portmux implementation defines
 * an enum containing all the possible peripheral functions that can
 * be selected. For example, the PIO implementation, which allows
 * multiplexing two peripherals per pin, defines it like this:
 *
 * \code
 * enum portmux_function {
 *	PORTMUX_FUNC_A,
 *	PORTMUX_FUNC_B,
 * };
 * \endcode
 *
 * To configure a set of pins to be connected to a given peripheral
 * function, the following function is used.
 *
 * \code
 * void portmux_select_peripheral(void *port, pin_mask_t pin_mask,
 *		enum portmux_function func, unsigned long flags);
 * \endcode
 *
 * To configure a set of pins to be controlled by software (GPIO), the
 * following function is used. In this case, no "function" argument is
 * required since "GPIO" is a function in its own right.
 *
 * \code
 * void portmux_select_gpio(void *port, pin_mask_t pin_mask,
 *		unsigned long flags);
 * \endcode
 *
 * Both of these functions take a "flags" parameter which may be used
 * to alter the default configuration of the pin. This is a bitmask of
 * various flags defined in an implementation-specific way, but the
 * names of the flags are the same on all implementations.  
 *
 * \subsection direction-flags Flags indicating pin direction
 *
 * The following flags specify the direction of a GPIO pin. They are
 * mutually exclusive and only valid when setting up a pin for
 * software control (GPIO):
 *
 * - \c PORTMUX_DIR_OUTPUT: Output pin. Setting this flag enables the
 *   output driver.
 * - \c PORTMUX_DIR_INPUT: Input pin. Setting this flag disables the
 *   output driver.
 *
 * \subsection init-state-flags Flags indicating initial pin state
 *
 * The following flags specify the initial state of a GPIO output pin.
 * They are mutually exclusive and only valid when setting up a GPIO
 * pin with the PORTMUX_DIR_OUTPUT flag set.
 *
 * - \c PORTMUX_INIT_HIGH: Drive the pin high (Vdd) initially.
 * - \c PORTMUX_INIT_LOW: Drive the pin low (Vss) initially.
 *
 * \subsection pull-flags Flags indicating pull-up/pull-down state
 *
 * The following flags specify the state of the on-chip CMOS resistors
 * connected to the pins. They are mutually exclusive and valid
 * regardless of any other pin configuration properties.
 *
 * Note that some portmux implementations may not support all of these
 * flags. If PORTMUX_BUSKEEPER is specified but not supported by the
 * hardware, the driver may enable a pull-up instead. If
 * PORTMUX_PULL_DOWN is specified but not supported by the hardware,
 * it will be ignored.
 *
 * - \c PORTMUX_PULL_UP: Weak pull-up to Vdd
 * - \c PORTMUX_PULL_DOWN: Weak pull-down to Vss
 * - \c PORTMUX_BUSKEEPER: Enable pull-up or pull-down depending on the
 *   state of the pin when it is released. This will prevent the pin
 *   from toggling.
 *
 * \subsection drive-strength-flags Flags indicating drive strength
 *
 * The following mutually-exlusive flags determine the drive strength of the
 * pins, if supported. If the hardware doesn't support configurable
 * drive strength, these flags are ignored.
 *
 * - \c PORTMUX_DRIVE_MIN: Minimum drive strength. This will give low
 *   power consumption, but may cause corruption of high-speed signals.
 * - \c PORTMUX_DRIVE_LOW: Low drive strength. Somewhat higher power
 *   consumption, but tolerates higher speed.
 * - \c PORTMUX_DRIVE_HIGH: High drive strength. Higher power
 *   consumption, can handle higher speeds.
 * - \c PORTMUX_DRIVE_MAX: Maximum drive strength. Highest power
 *   consumption, but can handle very high-speed signals.
 *
 * Note that setting the drive strength too high may cause excessive
 * overshoot and EMI problems, which may in turn cause signal
 * corruption.
 *
 * \subsection open-drain-flag Flags for supporting open-drain busses
 *
 * The following flag will configure the pins as "open drain", i.e.
 * setting the pin state to 0 will drive it low, while setting it to 1
 * will leave it floating (in most cases, an external or internal
 * pull-up resistor will ensure it goes high.)
 *
 * This mode is also known as "multi-drive" mode.
 *
 * - \c PORTMUX_OPEN_DRAIN
 *
 * \section portmux-peripherals Enabling specific peripherals
 *
 * In addition to the above functions, each chip provides a set of
 * functions for setting up the port multiplexer to use a given
 * peripheral. The following functions are available, if the selected
 * chip supports the corresponding feature:
 *
 * - portmux_enable_ebi(): \copybrief portmux_enable_ebi()
 * - portmux_enable_usart(): \copybrief portmux_enable_usart()
 * - portmux_enable_usbb(): \copybrief portmux_enable_usbb()
 * - portmux_enable_macb(): \copybrief portmux_enable_macb()
 * - portmux_enable_mmci(): \copybrief portmux_enable_mmci()
 * - portmux_enable_spi(): \copybrief portmux_enable_spi()
 *
 * All of these functions take a \a drive_strength parameter, which
 * must be one of the \c PORTMUX_DRIVE_x flags specified above.  Any
 * other portmux flags will be silently filtered out.
 *
 * \section using-gpio Using GPIO
 *
 * Once a pin has been set up as a GPIO, you can, depending on how it
 * was configured, drive it to a specific state or read the current
 * state of the pin. The following functions are available for this
 * purpose:
 *
 * \code
 * void gpio_set_value(unsigned int pin, bool value);
 * bool gpio_get_value(unsigned int pin);
 * \endcode
 */
