/**
 * \file
 *
 * \brief DMA mapping for AVR32 UC3
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 UC3 devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#ifndef CPU_DMA_H_INCLUDED
#define CPU_DMA_H_INCLUDED

#include <types.h>

/** \brief log2 of the minimum alignment of DMA-able objects */
#define CPU_DMA_ALIGN		2

/**
 * \brief A physical/virtual address pair used in struct buffer
 *
 * Since physical and virtual addresses are always identical on AVR32
 * UC3, they can share the same storage as a union.
 */
typedef union {
	phys_addr_t	phys;	//!< Physical address
	void		*ptr;	//!< Virtual address
} buffer_addr_t;

/**
 * \brief Synchronize a DMA buffer before transfer
 *
 * After this function has been called, the device can access the
 * buffer, but the CPU must not.
 *
 * \param vaddr The virtual address of the buffer
 * \param size The length of the buffer in bytes
 * \param direction The direction of the transfer
 */
static inline void dma_sync_for_device(void *vaddr, size_t size,
		enum dma_direction direction)
{

}

/**
 * \brief Synchronize a DMA buffer after transfer
 *
 * After this function has been called, the CPU can access the buffer,
 * but the device must not.
 *
 * \param vaddr The virtual address of the buffer
 * \param size The length of the buffer in bytes
 * \param direction The direction of the transfer
 */
static inline void dma_sync_for_cpu(void *vaddr, size_t size,
		enum dma_direction direction)
{

}

/**
 * \brief Map a single contiguous buffer for DMA transfer
 *
 * After this function has been called, the device can access the
 * buffer, but the CPU must not.
 *
 * \param vaddr The virtual address of the buffer
 * \param size The length of the buffer in bytes
 * \param direction The direction of the transfer
 *
 * \return The physical address of the buffer
 */
static inline phys_addr_t dma_map_single(void *vaddr, size_t size,
		enum dma_direction direction)
{
	return (phys_addr_t)vaddr;
}

/**
 * \brief Unmap a single contiguous buffer after a DMA transfer
 *
 * After this function has been called, the CPU can access the buffer,
 * but the device must not.
 *
 * \param vaddr The virtual address of the buffer
 * \param size The length of the buffer in bytes
 * \param direction The direction of the transfer
 *
 * \return The physical address of the buffer
 */
static inline void dma_unmap_single(void *vaddr, size_t size,
		enum dma_direction direction)
{

}

#endif /* CPU_DMA_H_INCLUDED */
