/**
 * \file
 *
 * Copyright (C) 2009 Atmel Corporation. All rights reserved.
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/*!
 * \file
 *
 * \brief AVR32-specific implementation of bit operations
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC v4.2+ for AVR32
 * - Supported devices:  All AVR32 devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Copyright (C) 2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef ARCH_BITOPS_H_INCLUDED
#define ARCH_BITOPS_H_INCLUDED

#include <compiler.h>

/**
 * \brief Atomically set bit \a nr in \a bitmap
 */
static inline void atomic_set_bit(unsigned int nr, unsigned long *bitmap)
{
	__sync_or_and_fetch(&bitmap[bit_word(nr)], bit_mask(nr));
}

/**
 * \brief Atomically clear bit \a nr in \a bitmap
 */
static inline void atomic_clear_bit(unsigned int nr, unsigned long *bitmap)
{
	__sync_and_and_fetch(&bitmap[bit_word(nr)], ~bit_mask(nr));
}

/**
 * \brief Atomically toggle bit \a nr in \a bitmap
 */
static inline void atomic_toggle_bit(unsigned int nr, unsigned long *bitmap)
{
	__sync_xor_and_fetch(&bitmap[bit_word(nr)], bit_mask(nr));
}

/**
 * \brief Atomically set bit \a nr in \a bitmap and return its
 * previous state.
 */
static inline bool atomic_test_and_set_bit(unsigned int nr,
		unsigned long *bitmap)
{
	unsigned long tmp;

	tmp = __sync_fetch_and_or(&bitmap[bit_word(nr)], bit_mask(nr));

	return 1UL & (tmp >> (nr & (BITS_PER_LONG - 1)));
}

/**
 * \brief Atomically clear bit \a nr in \a bitmap and return its
 * previous state.
 */
static inline bool atomic_test_and_clear_bit(unsigned int nr,
		unsigned long *bitmap)
{
	unsigned long tmp;

	tmp = __sync_fetch_and_and(&bitmap[bit_word(nr)], ~bit_mask(nr));

	return 1UL & (tmp >> (nr & (BITS_PER_LONG - 1)));
}

/**
 * \brief Find First bit Set in \a word
 *
 * \note If \a word is 0, the result is undefined.
 *
 * \return The lowest number \a x such that (word & (1UL << \a x)) is
 * nonzero.
 */
static inline unsigned long __ffs(unsigned long word)
{
	return count_leading_zeroes(bit_reverse(word));
}

/**
 * \brief Find First Zero bit in \a word
 *
 * \note If \a word has all bits set, the result is undefined.
 *
 * \return The lowest number \a x such that (word & (1UL << \a x)) is
 * zero.
 */
static inline unsigned long ffz(unsigned long word)
{
	return __ffs(~word);
}

/**
 * \brief Find Last bit Set in \a word
 *
 * \note If \a word is zero, the result is undefined.
 *
 * \return The highest number \a x such that (word & (1UL << \a x)) is
 * nonzero.
 */
static inline int __fls(unsigned long word)
{
	return 31 - count_leading_zeroes(word);
}

/**
 * \brief Find Last bit Set in \a word (counting from 1)
 *
 * This function differs from __fls() in that the least significant bit
 * is position 1, and it returns a well-defined result when \a word is
 * 0.
 *
 * \return The highest number \a x such that (word & (1UL << (\a x -
 * 1))) is nonzero, or 0 if \a word is zero.
 */
static inline int fls(unsigned long word)
{
	if (word == 0)
		return 0;

	return __fls(word) + 1;
}

/**
 * \brief Find First bit Set in \a word (standard semantics)
 *
 * This function differs from __ffs() in that the least significant
 * bit is position 1, and it returns a well-defined result when \a
 * word is 0.
 *
 * \return The lowest number \a x such that (word & (1UL << (\a x -
 * 1))) is nonzero, or 0 if \a word is zero.
 */
static inline int ffs(unsigned long word)
{
	if (word == 0)
		return 0;
	return __ffs(word) + 1;
}

#endif /* ARCH_BITOPS_H_INCLUDED */
