/**
 * \file
 *
 * \brief Default exception handler for AVR32.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 * Atmel AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#include <compiler.h>
#include <debug.h>
#include <types.h>

/**
 * \brief Default unknown exception handler for AVR32.
 *
 * This function is called when an unexpected exception is triggered.
 * It currently does nothing.
 *
 * This function can be overridden by the application if it needs to
 * do something special.
 *
 * \param cause The value of the Exception Cause Register.
 * \param regs A snapshot of the register file at the time when the
 * exception was triggered.
 */
void __weak do_unknown_exception(unsigned int cause, uint32_t *regs)
{
	dbg_panic("Unhandled exception %u\n", cause);
	dbg_panic(" sr: %08x  pc: %08x  lr: %08x  sp: %08x r12: %08x\n",
			regs[15], regs[16], regs[0], regs[1], regs[2]);
	dbg_panic("r11: %08x r10: %08x  r9: %08x  r8: %08x\n",
			regs[3], regs[4], regs[5], regs[6]);
	dbg_panic(" r7: %08x  r6: %08x  r5: %08x  r4: %08x\n",
			regs[7], regs[8], regs[9], regs[10]);
	dbg_panic(" r3: %08x  r2: %08x  r1: %08x  r0: %08x\n",
			regs[11], regs[12], regs[13], regs[14]);
}
