/*This file is prepared for Doxygen automatic documentation generation.*/
//! \file *********************************************************************
//!
//! \brief This file manages the USB Host mouse control application.
//!
//! - Compiler:           IAR EWAVR and GNU GCC for AVR
//! - Supported devices:  AT90USB1287, AT90USB1286, AT90USB647, AT90USB646
//!
//! \author               Atmel Corporation: http://www.atmel.com \n
//!                       Support and FAQ: http://support.atmel.no/
//!
//! ***************************************************************************

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//_____  I N C L U D E S ___________________________________________________

#include "config.h"
#include "conf_usb.h"
#include "host_hid_task.h"
#include "modules/usb/host_chap9/usb_host_task.h"
#include "modules/usb/host_chap9/usb_host_enum.h"
#include "lib_mcu/usb/usb_drv.h"

//_____ M A C R O S ________________________________________________________


//_____ D E F I N I T I O N S ______________________________________________


#ifndef LOG_STR_CODE
#define LOG_STR_CODE(str)
#else
U8 code log_enter_boot[]="Entering bootloader";
U8 code log_hid_connect[]="Generic HID connected";
#endif

//_____ D E C L A R A T I O N S ____________________________________________

U8 hid_connected;
#if( HID_GENERIC_DEMO_FULL == ENABLE )         
S8 hid_temperature=0;
U8 hid_potentiometer=0;
#endif
U8 hid_enter_dfu;
static U8 pipe_hid_in;
static U8 pipe_hid_out;


//!@brief This function initializes the host generic HID application
//!
//!
//!@param none
//!
//!@return none
void host_hid_task_init(void)
{
   hid_enter_dfu=0;
   hid_connected=0;
}

//! @brief This function manages the host generic HID application
//!
//! @param none
//!
//! @return none
void host_hid_task(void)
{
   U8 i;
   U8 buf_hid[8];
   
   // Test HID device connection
   if(Is_host_ready())
   {
      if(Is_new_device_connection_event())   //Device connection
      {
            hid_connected=0;
            hid_enter_dfu=0;
            for(i=0;i<Get_nb_supported_interface();i++)
            {
               if(Get_class(i)==HID_CLASS && Get_subclass(i)==NO_SUBCLASS && Get_protocol(i)==NO_PROTOCOL)
               {
                  LOG_STR_CODE(log_hid_connect);
                  hid_connected=1;
                  if(Is_ep_addr_in(Get_ep_addr(i,0)))
                  {
                     PIPE_HID_IN=host_get_hwd_pipe_nb(Get_ep_addr(i,0));
                     PIPE_HID_OUT=host_get_hwd_pipe_nb(Get_ep_addr(i,1));
                  }
                  else
                  {
                     PIPE_HID_IN=host_get_hwd_pipe_nb(Get_ep_addr(i,1));
                     PIPE_HID_OUT=host_get_hwd_pipe_nb(Get_ep_addr(i,0));
                  }
                  Host_select_pipe(PIPE_HID_IN);
                  Host_continuous_in_mode();
                  Host_unfreeze_pipe();
                  break;
               }
            }
      }
      if(hid_connected)
      {
         Host_select_pipe(PIPE_HID_IN);
         if(Is_host_in_received())
         {
            if(Is_host_stall()==FALSE)
            {
#if( HID_GENERIC_DEMO_FULL == ENABLE )         
               i=Usb_read_byte();                     // Leds
               i=Usb_read_byte();                     // Joystick
               hid_potentiometer=Usb_read_byte();     // Potentiometer
               hid_temperature=Usb_read_byte();       // Temperature
#else
               i=Usb_read_byte();                     // Joystick
#endif
            }
            Host_ack_in_received(); Host_send_in();
         }
         if(hid_enter_dfu==TRUE)
         {
            LOG_STR_CODE(log_enter_boot);
#if( HID_GENERIC_DEMO_FULL == ENABLE )         
            buf_hid[3]=0x55;
            buf_hid[4]=0xAA;
            host_send_data(PIPE_HID_OUT, 8, buf_hid);
#else
            buf_hid[0]=0x55;
            buf_hid[1]=0xAA;
            buf_hid[2]=0x55;
            buf_hid[3]=0xAA;
            usb_request.bmRequestType = USB_SETUP_SET_CLASS_INTER;
            usb_request.bRequest      = SETUP_HID_SET_REPORT;
            usb_request.wValue        = (REPORT_TYPE_FEATURE<<8);
            usb_request.wIndex        = 0;
            usb_request.wLength       = 4;
            usb_request.uncomplete_read = FALSE;
            host_send_control(buf_hid);
#endif
         }
      }
   }
   //Device disconnection...
   if(Is_device_disconnection_event())
   {
      Leds_off();
      hid_connected=0;
      hid_enter_dfu=0;
   }
}
