/*This file is prepared for Doxygen automatic documentation generation.*/
//! \file *********************************************************************
//!
//! \brief This file contains the low level macros and definition for stk600 board
//!
//! - Compiler:           IAR EWAVR and GNU GCC for AVR
//! - Supported devices:  ATmega32U4
//!
//! \author               Atmel Corporation: http://www.atmel.com \n
//!                       Support and FAQ: http://support.atmel.no/
//!
//! ***************************************************************************

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef STK_600_H
#define STK_600_H

//_____ I N C L U D E S ____________________________________________________
#include "config.h"

#if (TARGET_BOARD==STK600)

//_____ M A C R O S ________________________________________________________

//! @defgroup STK_600_module STK600
//! This module contains low level hardware abstraction layer for STK600 board with ATMega32U4.
//! Routing card: STK600-RC044U-25
//! Socket card: STK600-TQFP44
//!
//! The STK600 software configuration (use AVRStudio) :
//! - HW Settings/Clock Generator/8000MHz
//!
//! The STK600 hardware configuration :
//! - set clock switch on EXT clock (to use Clock generator from STK600)
//! - connect PORTB to SWITCHES
//! - connect PORTC to LEDS
//!
//! This configuration enables following interface switches 0 to 7 and leds 6 and 7.
//! @image html stk_600.jpg
//! @{


      //! @defgroup STK_600_leds Leds Management
      //! Macros to manage Leds on stk600
      //! @{
      
#define  LED_PORT             PORTC
#define  LED_DDR              DDRC
#define  LED_PIN              PINC
#define  LED6_BIT             PINC6
#define  LED7_BIT             PINC7                      

#define  Leds_init()          (LED_DDR  |=  (1<<LED7_BIT) | (1<<LED6_BIT))
#define  Leds_on()            (LED_PORT &= ~((1<<LED7_BIT) | (1<<LED6_BIT)))
#define  Leds_off()           (LED_PORT |=  (1<<LED7_BIT) | (1<<LED6_BIT))

#define  Led6_on()            (LED_PORT &= ~(1<<LED6_BIT))
#define  Led7_on()            (LED_PORT &= ~(1<<LED7_BIT))
#define  Led6_off()           (LED_PORT |=  (1<<LED6_BIT))
#define  Led7_off()           (LED_PORT |=  (1<<LED7_BIT))
#define  Led6_toggle()        (LED_PIN  |=  (1<<LED6_BIT))
#define  Led7_toggle()        (LED_PIN  |=  (1<<LED7_BIT))
#define  Is_led6_on()         (LED_PIN  &   (1<<LED6_BIT) ? FALSE : TRUE)
#define  Is_led7_on()         (LED_PIN  &   (1<<LED7_BIT) ? FALSE : TRUE)
      //! @}

      //! @defgroup STK_600_joy Joystick Management
      //! Macros to manage Joystick on stk600
      //! @{
#define  Switches_init()      (DDRB &= ~0xFF, PORTB |= 0xFF)
#define  Is_switch_7()        ((PINB & (1<<PINB7)) ?  FALSE : TRUE)
#define  Is_switch_6()        ((PINB & (1<<PINB6)) ?  FALSE : TRUE)
#define  Is_switch_5()        ((PINB & (1<<PINB5)) ?  FALSE : TRUE)
#define  Is_switch_4()        ((PINB & (1<<PINB4)) ?  FALSE : TRUE)
#define  Is_switch_3()        ((PINB & (1<<PINB3)) ?  FALSE : TRUE)
#define  Is_switch_2()        ((PINB & (1<<PINB2)) ?  FALSE : TRUE)
#define  Is_switch_1()        ((PINB & (1<<PINB1)) ?  FALSE : TRUE)
#define  Is_switch_0()        ((PINB & (1<<PINB0)) ?  FALSE : TRUE)
#define  Is_not_switch_7()    ((PINB & (1<<PINB7)) ?  TRUE : FALSE)
#define  Is_not_switch_6()    ((PINB & (1<<PINB6)) ?  TRUE : FALSE)
#define  Is_not_switch_5()    ((PINB & (1<<PINB5)) ?  TRUE : FALSE)
#define  Is_not_switch_4()    ((PINB & (1<<PINB4)) ?  TRUE : FALSE)
#define  Is_not_switch_3()    ((PINB & (1<<PINB3)) ?  TRUE : FALSE)
#define  Is_not_switch_2()    ((PINB & (1<<PINB2)) ?  TRUE : FALSE)
#define  Is_not_switch_1()    ((PINB & (1<<PINB1)) ?  TRUE : FALSE)
#define  Is_not_switch_0()    ((PINB & (1<<PINB0)) ?  TRUE : FALSE)

//! Enable interrupts for switches (PINB0 to PINB7)
//! - Set up interrupts for switches
//! - clear flag
//! - enable interrupt
#define  Switches_enable_it() {  \
      PCMSK0 |= (1<<PCINT0)|(1<<PCINT1)|(1<<PCINT2)|(1<<PCINT3)|(1<<PCINT4)|(1<<PCINT5)|(1<<PCINT6)|(1<<PCINT7);\
      PCIFR |= (1<<PCIF0);       \
      PCICR |= (1<<PCIE0);}

//! Disable interrupts for switches (PINB0 to PINB7)
//! - disable interrupt
//! - clear flag
//! - Clear mask
#define  Switches_disable_it() { \
      PCICR  &= ~(1<<PCIE0);     \
      PCIFR  |= (1<<PCIF0);      \
      PCMSK0 &= ~((1<<PCINT0)|(1<<PCINT1)|(1<<PCINT2)|(1<<PCINT3)|(1<<PCINT4)|(1<<PCINT5)|(1<<PCINT6)|(1<<PCINT7));}
      //! @}

//! @}

#endif   // TARGET_BOARD==STK600

#endif  // STK_600_H
