
###################################################################################
#                                                                                 #
# File Name:    max2syn_script.tcl                                             	  #
#                                                                                 #
# Version:      1.0                                                               #
#                                                                                 #
# Summary:      This TK script as a simple Graphical User Interface to compile    #
#               any design using the new MAX+PLUS II Advanced Synthesis           #
#                                                                                 #
# Licensing:    This script is  pursuant to the following license agreement       #
#               (BY VIEWING AND USING THIS SCRIPT, YOU AGREE TO THE               #
#               FOLLOWING): Copyright (c) 2003 Altera Corporation, San Jose,      #
#               California, USA.  Permission is hereby granted, free of           #
#               charge, to any person obtaining a copy of this software and       #
#               associated documentation files (the "Software"), to deal in       #
#               the Software without restriction, including without limitation    #
#               the rights to use, copy, modify, merge, publish, distribute,      #
#               sublicense, and/or sell copies of the Software, and to permit     #
#               persons to whom the Software is furnished to do so, subject to    #
#               the following conditions:                                         #
#                                                                                 #
#               The above copyright notice and this permission notice shall be    #
#               included in all copies or substantial portions of the Software.   #
#                                                                                 #
#               THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,   #
#               EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   #
#               OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          #
#               NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       #
#               HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      #
#               WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      #
#               FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     #
#               OTHER DEALINGS IN THE SOFTWARE.                                   #
#                                                                                 #
#               This agreement shall be governed in all respects by the laws of   #
#               the State of California and by the laws of the United States of   #
#               America.                                                          #
#                                                                                 #
# Description:  This TK script can be used as a wrapper to the MAX+PLUS II        #
#               Advanced Synthesis Tool                                           #
#               The script allows you to set up the basic command-line arguments  #
#               and run each executable independently or as part of pre-defined   #
#               scripts.                                                          # 
#                                                                                 #
#               You can run this script from a command line by typing:            #
#                                                                                 #
#                     max2synw -g [<project_name>]                                #
#    or               max2synw --gui [<project_name>]                             #
#                                                                                 #
#                                                                                 #
###################################################################################

# Only max2synw exes can interpret this script
if [catch { init_tk }] {
	set msg "Usage: max2synw -g \[<project_name>\]"
	puts $msg
	catch { tk_messageBox -type ok -message $msg -icon error }
	exit -1
}


#############################################################################
## Libraries


# Set global variables and constants
# ----------------------------------

# List of supported Tools
set tool_list { qsyn mpii }
set predef_flows { full_compile }

# Tool EXE names
set exe_name(qsyn) max2syn
set exe_name(mpii) maxplus2
set exe_name(mpiiw) max2win

# Tool user names
set user_name(qsyn) "MAX+PLUS II Advanced Synthesis"
set user_name(mpii) "MAX+PLUS II Compiler"
set user_name(full_compile) "Full Compilation"

# Tool buttons
set buttons(qsyn) ""
set buttons(mpii) ""
set buttons(full) ""
set buttons(stop) ""
set buttons(new) ""

set exit_script 0

######################################## Global Options Section #################################################

######################################## Fitter Section #################################################

# ----------------------------------------------------------------
#
namespace eval max2synw {
#
# Description: Initialize all internal variables
#
# ----------------------------------------------------------------

	global max2syn

	variable app_name $max2syn(nameofexecutable)
	variable version  $max2syn(version)
	variable max2synw_path $max2syn(binpath)

	variable mainframe
	variable tool_pane						# holds the right hand side of paned window (for tool buttons)
	variable paned_window
	variable pane_window_frame
	variable log_window						# displays messages from execution of tools 
	variable toolbar_status 1
	variable flow_frame
	variable flow_menu_exists 0
	variable progmsg
	variable progval 0
	
	# Set the BWidget directory path.

	variable project_dir						# holds the path to project directory
	variable project_name						# holds the name of the project

	variable acf

	# Log max2synw commands

	# Maintain list of recent projects opened

	# Variables for holding settings

	variable flow_name						# stores the name of the flow

	variable default_editor "<none>"
	variable default_mp2_root "<none>"
	variable use_default_editor 0
	variable use_default_mp2_root 0
	variable maxplus2_exe_found 0

	variable error_count 0

	# link the other max2synw scripts 

	# if Max2Synw project settings file exists, then read the dlg settings from the file otherwise assign defaults

	set max2synw::flow_name "full_compile"
	set max2synw::project_dir ""
	set max2synw::project_name "<none>"

	if [info exists env(EDITOR)] {
		set default_editor $env(EDITOR)
		set use_default_editor 1
	}

	if [info exists env(MAXPLUS2_ROOT)] {
		set default_mp2_root $env(MAXPLUS2_ROOT)
		set use_default_mp2_root 1

		# Mdify the PATH environment variable
		if [info exists env(PATH)] {
			if { [string equal -nocase $tcl_platform(platform) "unix"] } {			
			append env(PATH) ":$default_mp2_root/bin"
			} else {
				append env(PATH) ";$default_mp2_root"
			}
		} else {
			set env(PATH) $max2synw::default_mp2_root
		}
	}

	if { ![catch { exec maxplus2 -v }] } {
		set maxplus2_exe_found 1
	}

}

# ----------------------------------------------------------------
#
proc max2synw::main {} {
#
# Description: Entry point
#
# ----------------------------------------------------------------
	
	global max2syn
	variable app_name
	variable project_name

	option add *TitleFrame.l.font {helvetica 11 bold italic}
	option add *LabelFrame.l.font {helvetica 10}

	wm withdraw .
	wm title . "$app_name"

	max2synw::create
	BWidget::place . 0 0 center
	wm deiconify .
	raise .
	focus -force .

}

# ----------------------------------------------------------------
#
proc max2synw::create { } {
#
# Description: Creates all the GUI elements
#
# ----------------------------------------------------------------

	variable mainframe
	variable tool_pane
	variable paned_window
	variable pane_window_frame

	# Menu creation
	max2synw::init_menus

	# Toolbar creation
	max2synw::init_toolbar

	# Create a Paned window
	set pane_window_frame [$mainframe getframe]
	set paned_window [PanedWindow $pane_window_frame.pw -side top]

	#Create file pane (left side) and create second pane for buttons and message window (Right Side)
	set tool_pane [$paned_window add -weight 2]

	# Populate File Pane and Tool Pane
	max2synw::populate_tool_pane

	#Pack Paned Window
	pack $paned_window -fill both -expand yes -padx 4 -pady 4


	# Pack the mainframe
	update idletasks

	#Make sure when the user destroys the application, the project settings are stored in max2synw project file
	bind $mainframe <Destroy> { max2synw::close_max2synw }
}

# ----------------------------------------------------------------
#
proc max2synw::close_max2synw { } {
#
# Description: Writes the project settings back to max2synw project file (max2synw_settings.txt)
#
# ----------------------------------------------------------------

	global exit_script 
	global pipe

	if { [is_project_open] } {
		export_assignment_files
		project_close
	}
	
	if [info exists pipe] {

		catch {close $pipe}
	}

	set exit_script 1
}	

# ----------------------------------------------------------------
#
proc max2synw::init_menus { } {
#
# Description: Initialize all the menus
#
# ----------------------------------------------------------------

	variable mainframe
	variable project_name
	variable version

	set menu_desc {
		"&File" all file 0 {
			{command "New &Project..." {} "Create a new project" {} -command max2synw::on_new_project_dlg }
			{command "Open P&roject..." {} "Open an existing project" {} -command max2synw::on_open_project_dlg }
			{command "&Close Project" {project_open_menu} "Close the current project" {} -command max2synw::on_close_project }
			{separator}
			{command "&New Text File" {default_editor_menu} "Create a new text file" {} -command max2synw::on_new_file }
			{command "&Open Text File..." {} "Open a text file" {} -command max2synw::on_open_file }
			{separator}
			{command "Opt&ions..." {} "Set user options" {} -command max2synw::on_open_prefer_dlg }
			{separator}
			{command "E&xit" {} "Exit max2synw" {} -command exit }
		}
		"&Assign" all options 0 {
			{separator}
			{command "&Add/Remove HDL File(s)..." {project_open_menu} "Add and/or remove HDL files to the project " {} -command max2synw::on_add_remove_file_dlg }
			{command "Family and &Synthesis Settings..." {project_open_menu} "Edit device family and  synthesis settings" {} -command max2synw::open_qsyn_opt_dlg}
		}
		"&Processing" all file 0 {
			{command "Start &Full Compilation" {project_open_menu} "Start Full Compilation" {} -command "max2synw::on_start_full_compile"}
			{separator}
			{command "Start MAX+PLUS II Advanced &Synthesis" {project_open_menu} "Start MAX+PLUS II Advanced Synthesis to perform logic synthesis and technology mapping on design" {} -command "max2synw::on_start_max2syn" }
			{command "Start MAX+PLUS II &Compilation" {mp2_cmp_exe_menu} "Start Compilation using MAX+PLUS II" {} -command "max2synw::on_start_maxplus2" }
			{command "&Stop Process" {compiling_menu} "Stop current process" {} -command "max2synw::stop_compile" }
			{separator}
			{command "Open MAX+PLUS II Advanced Synthesis &Report" {project_open_menu} "Open MAX+PLUS II Advanced Synthesis Report" {} -command "max2synw::open_rpt qsyn" }
			{command "Open MAX+PLUS II Compiler Repor&t" {mp2_cmp_exe_menu} "Open MAX+PLUS II Compiler Report" {} -command "max2synw::open_rpt qfit" }
			{separator}
			{command "Open &MAX+PLUS II Manager" {mp2_exe_menu} "Opens MAX+PLUS II Graphical User Interface" {} -command "max2synw::on_open_max2win" }
		}
		"&Help" all options 0 {
			{command "MAX+PLUS II Advanced Synthesis &Readme..." {} "Show MAX+PLUS II Advanced Synthesis Readme" {} -command max2synw::on_help}
			{command "MAX+PLUS II Advanced Synthesis &License Agreement..." {} "Show MAX+PLUS II Advanced Synthesis License Agreement" {} -command max2synw::on_license}
			{separator}
			{command "&About MAX+PLUS II Advanced Synthesis..." {} "About MAX+PLUS II Advanced Synthesis" {} -command max2synw::about_dlg}
		}
	}

	set status "Done"
	set mainframe [ MainFrame .mainframe \
			-height  600\
			-menu	$menu_desc \
			-textvariable status \
			-width 1000]

    $mainframe addindicator -text "$max2synw::version"

    pack $mainframe -fill both -expand yes

	if { !$max2synw::use_default_editor } {
		$mainframe setmenustate default_editor_menu disabled
	}

}

# ----------------------------------------------------------------
#
proc max2synw::init_toolbar { } {
#
# Description: Initializes the toolbar
#
# ----------------------------------------------------------------

	variable mainframe
	variable project_dir
	variable project_name
	variable max2synw_path
	variable flow_name
	variable flow_frame
	variable use_default_editor
	variable maxplus2_exe_found

	global buttons
	global tool_list
	global predef_flows
	global exe_name
	global user_name
	

	## Adding general toolbar

	set toolbar [$mainframe addtoolbar]
	set bbox [ButtonBox $toolbar.bbox -spacing 0 -padx 2 -pady 2]
	set buttons(new) [$bbox add -image [Bitmap::get ${max2synw::max2synw_path}/icons/new] \
        	-highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 2 -pady 2 \
			-command "max2synw::on_new_file" \
			-helptext "New Text File"]
	if { !$max2synw::use_default_editor } {
		$buttons(new) configure -state disabled
	}

	$bbox add -image [Bitmap::get ${max2synw::max2synw_path}/icons/open] \
        	-highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 2 -pady 2 \
			-command "max2synw::on_open_file" \
			-helptext "Open Text File"

	pack $bbox -side left -anchor w

	set sep [Separator $toolbar.sep -orient vertical]
	pack $sep -side left -fill y -padx 4 -anchor w

	set bbox [ButtonBox $toolbar.bbox3 -spacing 0 -padx 2 -pady 2]
	set buttons(setting) [$bbox add -image [Bitmap::get ${max2synw::max2synw_path}/icons/ase] \
        			-highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 2 -pady 2 \
					-command max2synw::on_settings_dlg \
			        -helptext "Family and Synthesis Settings"]
	pack $bbox -side left -anchor w

	set sep [Separator $toolbar.sep5 -orient vertical]
	pack $sep -side left -fill y -padx 4 -anchor w

	set bbox [ButtonBox $toolbar.bbox4 -spacing 0 -padx 2 -pady 2]
	set buttons(qsyn) [$bbox add -image [Bitmap::get ${max2synw::max2synw_path}/icons/syn_start] \
        			-highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 2 -pady 2 \
					-command "max2synw::on_start_max2syn" \
					-helptext "Start $user_name(qsyn)"]
	pack $bbox -side left -anchor w

	set buttons(mpii) [$bbox add -image [Bitmap::get ${max2synw::max2synw_path}/icons/mp2cmp] \
        			-highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 2 -pady 2 \
					-command "max2synw::on_start_maxplus2" \
					-helptext "Start $user_name(mpii)"]
	pack $bbox -side left -anchor w
	if { !$max2synw::maxplus2_exe_found } {
		$buttons(mpii) configure -state disabled
	}	

	set buttons(full) [$bbox add -image [Bitmap::get ${max2synw::max2synw_path}/icons/cmp_start] \
        			-highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 2 -pady 2 \
					-command "max2synw::on_start_$flow_name" \
					-helptext "Start Full Compilation"]
	pack $bbox -side left -anchor w

	set buttons(stop) [$bbox add -image [Bitmap::get ${max2synw::max2synw_path}/icons/stop] \
        			-highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 2 -pady 2 \
					-command "max2synw::stop_compile" \
					-helptext "Stop current process"]
	pack $bbox -side left -anchor w

	set buttons(mpiiw) [$bbox add -image [Bitmap::get ${max2synw::max2synw_path}/icons/mplus2] \
			-highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 1 -pady 1 \
			-command "max2synw::on_open_max2win" \
			-helptext "Open MAX+PLUS II Manager"]
	pack $bbox -side left -anchor w
	if { !$max2synw::maxplus2_exe_found } {
		$buttons(mpiiw) configure -state disabled
	}	

	set sep [Separator $toolbar.sep3 -orient vertical]
	pack $sep -side left -fill y -padx 4 -anchor w

	set bbox [ButtonBox $toolbar.bbox2 -spacing 0 -padx 2 -pady 2]
	set buttons(qsyn_rpt) [$bbox add -image [Bitmap::get ${max2synw::max2synw_path}/icons/syn_rpt] \
        			-highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 2 -pady 2 \
					-command "max2synw::open_rpt qsyn" \
			        -helptext "Open MAX+PLUS II Advanced Synthesis Report"]
	set buttons(mpii_rpt) [$bbox add -image [Bitmap::get ${max2synw::max2synw_path}/icons/mp2_rpt] \
        			-highlightthickness 0 -takefocus 0 -relief link -borderwidth 1 -padx 2 -pady 2 \
					-command "max2synw::open_rpt qfit" \
			        -helptext "Open MAX+PLUS II Compiler Report"]
	if { !$max2synw::maxplus2_exe_found } {
		$buttons(mpii_rpt) configure -state disabled
	}	

	pack $bbox -side left -anchor w

	set sep [Separator $toolbar.sep2 -orient vertical]
	pack $sep -side left -fill y -padx 4 -anchor w

	#Disable the buttons if no project is opened
	if { $max2synw::project_dir == "" } { 
		max2synw::set_compiler_status disabled
	}
}

# ----------------------------------------------------------------
#
proc max2synw::populate_tool_pane { } {
#
# Description: Populate the tool pane with the buttons and Message Window
#
# ----------------------------------------------------------------

	variable tool_pane
	
	#Create Log Window for displaying messages
	max2synw::init_log_window
}	

# ----------------------------------------------------------------
#
proc max2synw::init_log_window { } {
#
# Description:	Initialize application LOG Window
#
# ----------------------------------------------------------------

	global max2syn
	
	# Variables related to Application
	variable app_name
	variable version
	variable project_name

	# Variables related to Widgets
	variable mainframe
	variable log_window
	variable tool_pane
	variable default_editor
	variable default_mp2_root

	set label_frame	[LabelFrame $tool_pane.label_frame -text "Message Log Window:" -font {helvetica 10 bold italic} \
							   -side top \
							   -anchor nw \
							   -relief sunken \
							   -borderwidth 1]

	set sub_frame [$label_frame getframe]
	set log_window [text $sub_frame.log_window -setgrid false -fg black -font {Courier}  \
						   -yscrollcommand "$sub_frame.y_scroll set"]
	scrollbar $sub_frame.y_scroll -orient vertical -command "$sub_frame.log_window yview"

	pack $sub_frame.y_scroll -side right -fill y 
	pack $sub_frame.log_window -side left -expand yes -fill both
	pack $sub_frame -side top -fill both -expand yes 

	pack $label_frame -side  top -fill both -expand yes

	# Creating tags to control display of message like color, font, size etc.
	$log_window tag configure boldtag -font {Courier 10 bold}
	$log_window tag configure errortag -foreground red -font {Courier 10}
	$log_window tag configure warningtag -foreground blue -font  {Courier 10}
	$log_window tag configure infotag -font {Courier 10}

	max2synw::print_msg "---------------------------------------------------------------------------------------"
	max2synw::print_msg ""
	if { [string match $max2synw::default_editor "<none>"] } {
		max2synw::print_warning "Default text editor has not been specified."
		max2synw::print_warning "Text files will be displayed (read-only) in the Message Log Window."
		max2synw::print_warning "To open and edit text files, set the EDITOR environment variable to a text editor executable"
	} else {
		max2synw::print_msg "Found environment variable EDITOR=$max2synw::default_editor"
		max2synw::print_msg "Text files will be opened using $max2synw::default_editor <file_name>"
	}
	max2synw::print_msg ""
	if { [string match $max2synw::default_mp2_root "<none>"] } {
		max2synw::print_warning "MAX+PLUS II root has not been specified."
		max2synw::print_warning "To specify the location of the MAX+PLUS II executables, set the MAXPLUS2_ROOT environment variable."
		max2synw::print_msg ""
		if { $max2synw::maxplus2_exe_found } {
			max2synw::print_msg "MAX+PLUS II executables have been found in the PATH"
		} else {
			max2synw::print_error "MAX+PLUS II executables have not been found in the PATH"
			max2synw::print_error "MAX+PLUS II Advanced Synthesis Software will not be able to start the MAX+PLUS II Compiler and/or Manager"
		}
	} else {
		max2synw::print_msg "Found environment variable MAXPLUS2_ROOT=$max2synw::default_mp2_root"
		if { [file exists $max2synw::default_mp2_root] } {
			if { ![file isdirectory $max2synw::default_mp2_root] } {
				max2synw::print_error "$max2synw::default_mp2_root is not a directory path"
			} else {
				if { $max2synw::maxplus2_exe_found } {
					max2synw::print_msg "MAX+PLUS II executables have been found"
				} else {
					max2synw::print_error "MAX+PLUS II executables have not been found"
					max2synw::print_error "MAX+PLUS II Advanced Synthesis Software will not be able to start the MAX+PLUS II Compiler and/or Manager"
				}
			}
		} else {
			max2synw::print_error "$max2synw::default_mp2_root does not exist"
		}
	}
	max2synw::print_msg ""
	max2synw::print_msg "---------------------------------------------------------------------------------------"
	max2synw::print_msg ""

	# If we got an argument, assume it is the project_name
	set argc [llength $max2syn(args)]
	if { $argc == 0 } {
		set max2synw::project_name "<none>"
	} else {
		if { $argc == 1 } {
			set max2synw::project_name [lindex $max2syn(args) 0] 
			if { [project_exists $max2synw::project_name] } {
				max2synw::open_project
			} else {
				max2synw::create_project
			}
		}
	}
}

########### Functions for Menu items########################################################################################

# ----------------------------------------------------------------
#
proc max2synw::set_compiler_status { state } {
#
# Description:	Depending on the state, enable or disable all
#				releant buttons
#
# ----------------------------------------------------------------

	global buttons

	variable maxplus2_exe_found
	variable mainframe
	variable project_name

	switch $state {
		running { 
			# Process is running. 
			# Disable tool buttons
			set project_open_state disabled
			# Enable stop button
			set compiling_state normal
		}
		waiting { 
			# Process is not  running. 
			# Enable tool buttons
			set project_open_state normal
			# Disable stop button
			set compiling_state disabled
		}
		disabled { 
			# Project is not open
			# Disable tool buttons
			set project_open_state disabled
			# Disable stop button
			set compiling_state disabled
		}
		default {
			# Ilegal condition
			puts stderr "Internal Error: Illegal condition"
			exit -1
		}
	}
	

	# Enable all tool buttons and flow button after a Project is Selected
	$buttons(qsyn) configure -state $project_open_state
	# Show M+II related menus/buttons only if EXE is found
	if { $max2synw::maxplus2_exe_found } {
		if { [file exists ${max2synw::project_name}.edf] } {
			$buttons(mpii) configure -state $project_open_state
			$mainframe setmenustate mp2_cmp_exe_menu $project_open_state
		} else {
			# We don't want anybody compiling in M+II without an EDF
			# i.e. without first running Advance Synthesis
			$buttons(mpii) configure -state disabled
			$mainframe setmenustate mp2_cmp_exe_menu disabled
		}
		$buttons(full) configure -state $project_open_state
		$buttons(mpiiw) configure -state $project_open_state
		$buttons(mpii_rpt) configure -state $project_open_state
		$mainframe setmenustate mp2_exe_menu $project_open_state
	} else {
		$buttons(full) configure -state disabled
		$buttons(mpii) configure -state disabled
		$buttons(mpiiw) configure -state disabled
		$buttons(mpii_rpt) configure -state disabled
		$mainframe setmenustate mp2_exe_menu disabled
	}
	$buttons(setting) configure -state $project_open_state
	$buttons(qsyn_rpt) configure -state $project_open_state
	$buttons(stop) configure -state $compiling_state

	# Enable or disable Menus
	$mainframe setmenustate project_open_menu $project_open_state
	$mainframe setmenustate compiling_menu $compiling_state
}

# ----------------------------------------------------------------
#
proc max2synw::open_project { } {
#
# Description:	Open Project
#
# ----------------------------------------------------------------

	variable app_name

	max2synw::print_msg "Opening project: $max2synw::project_name"
	project_open $max2synw::project_name
	wm title . "$max2synw::app_name - $max2synw::project_name"

	# Enable all tool buttons and flow button after a Project is Selected
	max2synw::set_compiler_status waiting
}

# ----------------------------------------------------------------
#
proc max2synw::create_project { } {
#
# Description:	Create Project
#
# ----------------------------------------------------------------

	variable app_name
	variable project_name

	max2synw::print_msg "Creating project: $max2synw::project_name"
	project_new $max2synw::project_name				
	wm title . "$max2synw::app_name - $max2synw::project_name"

	# Enable all tool buttons and flow button after a Project is Selected
	max2synw::set_compiler_status waiting
}

# ----------------------------------------------------------------
#
proc max2synw::on_new_project_dlg {} {
#
# Description:	Dialog Box to create project
#
# ----------------------------------------------------------------

	variable project_dir
	variable project_name
	variable app_name

	global tool_list
	global predef_flows

	set typelist {
		{"Project Files" {".max2syn"}}
		{"All Files" {"*.*"}}
	}

	if { [is_project_open] } {
		set dlg_result [tk_messageBox -type yesno -message "Project is already open. OK to close first?" -title "$max2synw::app_name" -icon question]
		if { [string match $dlg_result "yes"] } {
			max2synw::on_close_project
		} else {
			# Exit without showing open DLG
			return
		}
	}

	set project_file_name [tk_getSaveFile -filetypes $typelist -title "New Project" -defaultextension .max2syn]
	if { $project_file_name == "" } {
		max2synw::print_msg "No project was created"
		max2synw::on_close_project
	} else {
		set max2synw::project_dir [file dirname $project_file_name]
		cd $max2synw::project_dir
		set max2synw::project_name [file rootname [file tail $project_file_name]]
		# Display files in tree structure in left window pane in files tab
	
		# Open/Create Project
		if { [project_exists $max2synw::project_name] } {	
			# Need to delete all project files first
			file delete -force db
			file delete -force $max2synw::project_name.psf
			file delete -force $max2synw::project_name.csf
			file delete -force $max2synw::project_name.esf
			file delete -force $max2synw::project_name.max2syn
		}

		if { [catch {max2synw::create_project}] } {
			
			max2synw::print_error "$max2synw::project_name is not a legal project name (check for illegal characters)"
			max2synw::on_close_project
		}
	}
}

# ----------------------------------------------------------------
#
proc max2synw::on_open_project_dlg {} {
#
# Description:	Dialog Box to open project
#
# ----------------------------------------------------------------

	variable project_dir
	variable project_name
	variable app_name

	global tool_list
	global predef_flows

	set typelist {
		{"Project Files" {".max2syn"}}
		{"Design Files" {".v" ".vhd" ".vhdl"}}
		{"MAX+PLUS II Files" {".acf"}}
		{"All Files" {"*.*"}}
	}

	if { [is_project_open] } {
		set dlg_result [tk_messageBox -type yesno -message "Project is already open. OK to close first?" -title "$max2synw::app_name" -icon question]
		if { [string match $dlg_result "yes"] } {
			max2synw::on_close_project
		} else {
			# Exit without showing open DLG
			return
		}
	}

	set project_file_name [tk_getOpenFile -filetypes $typelist -title "Open a Project"]
	if { $project_file_name == "" } {
		max2synw::print_msg "No project was open"
		max2synw::on_close_project
	} else {
		set max2synw::project_dir [file dirname $project_file_name]
		cd $max2synw::project_dir
		set max2synw::project_name [file rootname [file tail $project_file_name]]
		# Display files in tree structure in left window pane in files tab
	
		# Open/Create Project
		if { ![project_exists $max2synw::project_name] } {
			set dlg_result [tk_messageBox -type yesno -message "$max2synw::app_name Project does not exist. OK to create?" -title "$max2synw::app_name" -icon question]
			if { [string match $dlg_result "yes"] } {
				max2synw::create_project
			}
				
			# Exit without showing open DLG
			return
		}

		max2synw::open_project
	}
}

# ----------------------------------------------------------------
#
proc max2synw::on_new_file {} {
#
# Description:	Dialog Box to open a text file
#
# ----------------------------------------------------------------

	variable use_default_editor
	variable default_editor

	if { $max2synw::use_default_editor } {
		# Launch external editor
		max2synw::print_msg "Calling: $max2synw::default_editor"
		if [catch { exec $max2synw::default_editor & } result] {
			max2synw::print_error "Error: $result"
		}
	} else {
		max2synw::print_msg "Ignored command: No default editor is defined"
	}
}

# ----------------------------------------------------------------
#
proc max2synw::on_open_file {} {
#
# Description:	Dialog Box to open a text file
#
# ----------------------------------------------------------------


	set typelist {
		{"Design Files" {".v" ".vhd" ".vhdl"}}
		{"Project Files" {".max2syn" ".psf" ".csf"}}
		{"MAX+PLUS II Files" {".acf"}}
		{"All Files" {"*.*"}}
	}

	set file_name [tk_getOpenFile -filetypes $typelist -title "Open"]
	if { $file_name == "" } {
	} else {
		max2synw::open_ascii_file $file_name
	}
}

# ----------------------------------------------------------------
#
proc max2synw::on_add_button { lb } {
#
# Description: Use GetFile to get HDL file
#
# ----------------------------------------------------------------

	global tcl_platform

	set typelist {
		{"Design Files" {".v" ".vhd" ".vhdl"}}
	}

	set new_file_name [tk_getOpenFile -filetypes $typelist -title "Add file to project"]
	if { $new_file_name == "" } {
#		max2synw::print_msg "No file was added"
	} else {

		set num_entries [$lb size]
	
		# Check if file is already there
		for { set i 0 } { $i < $num_entries } { incr i } {
			set file [$lb get $i]
			if { [string equal -nocase $tcl_platform(platform) "unix"] } {
				if { [string equal "[file split $file]" "[file split $new_file_name]"] } {
					max2synw::print_msg "File $file is already in the project"
					return
				}
			} else {
				if { [string equal -nocase "[file split $file]" "[file split $new_file_name]"] } {
					max2synw::print_msg "File $file is already in the project"
					return
				}
			}
		}

		$lb insert end $new_file_name
	} 
}

# ----------------------------------------------------------------
#
proc max2synw::update_file_list { lb } {
#
# Description:	Update file list based on a new addition/deletion
#
# ----------------------------------------------------------------

	set file_list [get_global_assignment -name SOURCE_FILE]

	# Add each file to the list
	foreach file $file_list {
		$lb insert end $file
	}
}

# ----------------------------------------------------------------
#
proc max2synw::on_updown_button { lb updown } {
#
# Description:	Dialog Box to remove files to project
#
# ----------------------------------------------------------------

	set num_entries [$lb size]
	
	if { [llength [$lb curselection]] != 1 } {
		max2synw::print_error "Only one selection can be moved at one time"
		$lb selection clear 0 end
		return
	}

	for { set i 0 } { $i < $num_entries } { incr i } {
		set file [$lb get $i]
		set new($i) $file
	}

	for { set i 0 } { $i < $num_entries } { incr i } {
		set file [$lb get $i]
		if [$lb selection includes $i] {
			
			if { $updown == "up" } {
				if { $i > 0 } {
					set temp $new([expr $i - 1])
					set new([expr $i - 1]) $file
					set new($i) $temp
					set new_sel [expr $i - 1]
#					max2synw::print_msg "Swaping: $new([expr $i - 1]) -- $new($i)"
					break
				} else {
					# Do nothing
					set new_sel 0
				}
			} else {
				if { $i < [expr $num_entries - 1] } {
					set temp $new([expr $i + 1])
					set new([expr $i + 1]) $file
					set new($i) $temp
					set new_sel [expr $i + 1]
#					max2synw::print_msg "Swaping: $new($i) -- $new([expr $i + 1])"
					break
				} else {
					# Do nothing
					set new_sel [expr $num_entries - 1]
				}
			}
		}
	}

	$lb delete 0 end

	if [info exists new] {
		for { set i 0 } { $i < $num_entries } { incr i } {
			set file $new($i)
			$lb insert end $file
		}
	}
	if [info exists new_sel] {
		$lb selection set $new_sel
	}

}

# ----------------------------------------------------------------
#
proc max2synw::on_delete_button { lb } {
#
# Description:	Dialog Box to remove files to project
#
# ----------------------------------------------------------------

	foreach file_index [lsort -integer -decreasing [$lb curselection]] {
		$lb delete $file_index
	}
}

# ----------------------------------------------------------------
#
proc max2synw::on_add_remove_file_dlg {} {
#
# Description:	Dialog Box to and and/or remove files to project
#
# ----------------------------------------------------------------

	if { ![is_project_open] } {
		max2synw::print_error "No project is open"
		return 
	}

	set new_file ""

	set dlg [Dialog .add_remove_dlg -parent .\
				-modal  local \
				-separator 1 \
				-title "Add/Remove HDL files to project" \
				-side bottom \
				-anchor c \
				-default 0 \
				-cancel 1 ]

	$dlg add -name ok
	$dlg add -name cancel

	set sub_frame_1 [frame $dlg.sub_frame_1 -borderwidth 1 -relief groove]
	set sw    [ScrolledWindow $sub_frame_1.sw]
	set lb    [listbox $sw.lb -height 8 -width 50 -highlightthickness 0 -selectmode extended]

	$sw setwidget $lb

	max2synw::update_file_list $lb

	set file_new_frame_1 [LabelFrame $sub_frame_1.family_frame \
			-text "Current list of files:" -width 15 -font {helvetica 10}]
	pack $file_new_frame_1 -padx 7 -pady 7 -fill x

	pack $sw -fill both -expand yes
	pack $sub_frame_1 -padx 7 -pady 7

	set sub_frame_2 [frame $dlg.sub_frame_2 -borderwidth 1 -relief groove]
	set delete_button [Button $sub_frame_2.delete -text "Delete" \
			-command "max2synw::on_delete_button $lb"]
	set add_button [Button $sub_frame_2.add -text "Add..." \
			-command "max2synw::on_add_button $lb"]
	set down_button [Button $sub_frame_2.down -text "Down" \
			-command "max2synw::on_updown_button $lb down"]
	set up_button [Button $sub_frame_2.up -text "Up" \
			-command "max2synw::on_updown_button $lb up"]
	pack $add_button -side left
	pack $delete_button -side left
	pack $up_button -side left
	pack $down_button -side left
	pack $sub_frame_2 -padx 7 -pady 7

	set cancel	[ $dlg draw ]

	if { !$cancel } {
		# First delete all existing files
		foreach file [get_global_assignment -name SOURCE_FILE] {
			# Don't complain if we cannot remove it.
			catch {set_global_assignment -name SOURCE_FILE $file -remove}
		}
		max2synw::print_msg "List of Source Files:"
		set i 1
		foreach file_index [$lb get 0 end] {
			if { [catch {set_global_assignment -name SOURCE_FILE $file_index}] } {
				max2synw::print_error "Unable to add $file_index"
			} else {
				max2synw::print_msg "   $i : $file_index"
				incr i
			}
		}
		if { $i == 1 } {
			# No files exist
				max2synw::print_msg "   <none>"
		}
		max2synw::print_msg ""
		export_assignment_files
	}

	destroy $dlg

}

# ----------------------------------------------------------------
#
proc max2synw::on_close_project {} {
#
# Description:	Close Project
#
# ----------------------------------------------------------------

	variable project_name

	if { [is_project_open] } {
		max2synw::print_msg "Closing project: $max2synw::project_name"
		export_assignment_files
		project_close

	} else {
		max2synw::print_error "No project is open"
	}
	
	# Make sure everything is disabled
	wm title . "$max2synw::app_name"
	max2synw::set_compiler_status disabled
		
}

# ----------------------------------------------------------------
#
proc max2synw::on_open_prefer_dlg {} {
#
# Description:	Dialog Box to open user preferences
#
# ----------------------------------------------------------------

	variable default_editor
	variable default_mp2_root
	variable use_default_editor
	variable use_default_mp2_root
	global buttons
	variable mainframe

	set editor $max2synw::default_editor
	set mp2_root $max2synw::default_mp2_root

	set dlg [Dialog .prefer_dlg -parent .\
				-modal  local \
				-separator 1 \
				-title "Options" \
				-side bottom \
				-anchor c \
				-default 0 \
				-cancel 1 ]

	$dlg add -name ok

	set frame [$dlg getframe]

	set default_frame [frame $dlg.defaut_frame -borderwidth 1 -relief groove -width 55]
	set editor_chk [checkbutton $default_frame.editor_chk -text "Use EDITOR environment variable:" -font {helvetica 10} -width 20 -anchor w \
                  -variable max2synw::use_default_editor \
				  -width 55 \
                  -onvalue  1 -offvalue 0]
	set editor_box [Label $default_frame.editor_entry -text "Value: $editor" -width 60]
	if { [string match $editor "<none>"] } {
		$editor_chk configure -state disabled
		$editor_box configure -state disabled
	} else {
		$editor_chk configure -state normal
		$editor_box configure -state normal
	}


 	grid $editor_chk -padx 7 -pady 1 -sticky w
 	grid $editor_box -padx 7 -pady 1 -sticky w
	pack $default_frame -padx 9 -pady 6 -side top

	pack $frame
	set res [$dlg draw]

	destroy $dlg

	if { !$max2synw::use_default_editor } {
		$mainframe setmenustate default_editor_menu disabled
		$buttons(new) configure -state disabled
	} else {
		$mainframe setmenustate default_editor_menu normal
		$buttons(new) configure -state normal
	}
}

# ----------------------------------------------------------------
#
proc max2synw::print_error { msg } {
#
# Description:	Display error in Log window
#				
#
# ----------------------------------------------------------------

	variable log_window

	$log_window insert end ">> Error: $msg\n" errortag

	$log_window yview moveto 10
}

# ----------------------------------------------------------------
#
proc max2synw::print_warning { msg } {
#
# Description:	Display warning in Log window
#				
#
# ----------------------------------------------------------------

	variable log_window

	$log_window insert end ">> Warning: $msg\n" warningtag

	$log_window yview moveto 10
}

# ----------------------------------------------------------------
#
proc max2synw::print_cmd { command } {
#
# Description:	Display command in log window and store in
#				log db
#				
#
# ----------------------------------------------------------------

	variable log_window

	$log_window insert end ">> $command\n" boldtag

	$log_window yview moveto 10
}

# ----------------------------------------------------------------
#
proc max2synw::log_reader { pipe } {
#
# Description:	On fileevent (executable log), we add the log
#				message to the LOG window
#
# ----------------------------------------------------------------

	variable log_window
	global done_reading
	variable error_count

	if { [eof $pipe] } {
		catch { close $pipe }
		set done_reading 1
		return
	}


	if { [gets $pipe line] < 0 } {
		return
	} else {
		set not_done 1
#		while { $not_done } {
			if [string match "Error:*" $line] {
				$log_window insert end "$line\n" errortag
				incr max2synw::error_count
			} elseif [string match "Internal Error:*" $line] {
				$log_window insert end "$line\n" errortag
				incr max2synw::error_count
			} elseif [string match "Warning: This computer program is protected*" $line] {
				# This is kind of hack to filter out first M+II warning
				$log_window insert end "$line\n" infotag
			} elseif [string match "Warning:*" $line] {
				$log_window insert end "$line\n" warningtag
			} elseif [string match "Info:*" $line] {
				$log_window insert end "$line\n" infotag
			} else {
				# Default to Info
				$log_window insert end "$line\n" infotag
			}
#			set not_done [expr [gets $pipe line] > 0]
#		}
		
		$log_window yview moveto 10000
	}
#	update
}

# ----------------------------------------------------------------
#
proc max2synw::launch_tool { tool } {
#
# Description:	Figure out required arguments and call 
#				require stand-alone executable
#
# ----------------------------------------------------------------
	
	variable log_window
	global done_reading
	variable project_name
	variable pids
	variable error_count
	variable default_mp2_root
	variable use_default_mp2_root
	variable max2synw_path
	global tool_list
	global predef_flows
	global exe_name
	global user_name
	global env
	global pipe

	#Reset error count before running EXE	
	set max2synw::error_count 0

	#Disable all tools
	max2synw::set_compiler_status running

	set done_reading 0

	set tool_command ""
	
	switch $tool {
		qsyn { 
			set tool_command "$exe_name(qsyn) $project_name" 
		}
		mpii { 
			set tool_command "$exe_name(mpii) -c ${project_name}.edf" 
		}
		default { 
			max2synw::print_error "Error: Unsupported Tool $exe_name($tool)"
			set tool_command ""
		
		}
	}
	max2synw::print_cmd $tool_command
	max2synw::print_cmd ""
	update

	set pipe [open "|$tool_command" r]
	set max2synw::pids [pid $pipe]
#	max2synw::print_msg "PID = $pids"
	fconfigure $pipe -blocking 0
	fileevent $pipe readable [list max2synw::log_reader $pipe]

	vwait done_reading

	# Enable all buttons after execution
	max2synw::set_compiler_status waiting

	return $max2synw::error_count
}




##########Functions for Toolbar items########################################################################################

# ----------------------------------------------------------------
#
proc max2synw::open_ascii_file { file_name } {
#
# Description:	Open an ASCII file with the user editor or
#		the message log if no editor is available
#
# ----------------------------------------------------------------

	variable log_window
	variable project_name
	variable default_editor
	variable use_default_editor
		
	max2synw::print_msg "Opening text file: $file_name"
	max2synw::print_msg ""
	if [file exists $file_name] {
		if { $max2synw::use_default_editor } {
			# Launch external editor
			max2synw::print_msg "Calling: $max2synw::default_editor $file_name"
			if [catch { exec $max2synw::default_editor $file_name & } result] {
				max2synw::print_error "Error: $result"
			}
			max2synw::print_msg ""
		} else {
			if [catch {open $file_name r} ascii_file] {
				max2synw::print_error "Error: Cannot open $file_name"
			} else {
				max2synw::print_msg "File Name: $file_name"
				max2synw::print_msg "------------------------------------------"
				max2synw::print_msg ""

				# If no editor is specified, read and display in Log window
				while {([gets $ascii_file ascii_line] >= 0)} {

					$log_window insert end "$ascii_line\n" infotag
				}	
				max2synw::print_msg "------------------------------------------"
				max2synw::print_msg ""
				close $ascii_file
				$log_window yview moveto 2000
			} 
		}
	} else {
		max2synw::print_error "Error: Cannot find $file_name"
	}
}

# ----------------------------------------------------------------
#
proc max2synw::open_rpt { tool } {
#
# Description:	Open RPT file for the given tool
#
# ----------------------------------------------------------------

	variable log_window
	variable project_name
	variable default_editor
	variable use_default_editor
		
	if { ![is_project_open] } {
		max2synw::print_error "No project is open"
		return
	}

	switch $tool {
		qsyn { 
			set rpt_file_name $project_name.syn.rpt
		}
		qfit { 
			set rpt_file_name $project_name.rpt
		}
		default { 
			max2synw::print_error "Error: Unsupported Tool $tool_name"
		}
	}

	max2synw::open_ascii_file $rpt_file_name
}

# ----------------------------------------------------------------
#
proc max2synw::print_msg { msg } {
#
# Description:	Display message in either log box (for GUI)
#				or using "puts" (for cmd)
#				
#				If cerr, also use puts to send to stderr
#
# ----------------------------------------------------------------

	variable log_window

	$log_window insert end ">> $msg\n" infotag

	$log_window yview moveto 10
}

###Progress Bar Dialog#############################
# ----------------------------------------------------------------
#
proc max2synw::stop_compile { } {
#
# Description:	On fileevent (executable log), we add the log
#				message to the LOG window
#
# ----------------------------------------------------------------

	variable pids
	variable error_count
	global tool_list
	global pipe

	if { ![is_project_open] } {
		max2synw::print_error "No project is open"
		return
	}

#	max2synw::print_msg "Killing PID = $max2synw::pids"
	set max2synw::error_count -1

	catch { exec kill $pids }

	set done_reading 1
	
	max2synw::print_msg ""
	max2synw::print_msg "Processed Stopped"
	max2synw::print_msg ""

	# Enable all buttons after execution
	max2synw::set_compiler_status waiting

}

############Functions for handling tool buttons###########################################################################

# ----------------------------------------------------------------
#
proc max2synw::set_acf { } {
#
# Description:	Use M+II's setacf command to propagete user settings
#
# ----------------------------------------------------------------
	
	variable project_name
	variable max2synw_path
	global exe_name
	global user_name

	set acf_name ${max2synw::project_name}.acf

	# If non exist, make a new ACF
	if { ![file exists $acf_name ] } {
		max2synw::print_msg "Creating ACF: $acf_name"
		if { [catch { exec setacf -c -f$acf_name }] } {
			print_msg "No M+II found. Skiping ACF creating/editing"
			return
		}
	} else {
		max2synw::print_msg "Modifying ACF: $acf_name"
	}

	# And now we get our settings
	if { [catch { exec setacf -f$acf_name -sGLOBAL_PROJECT_SYNTHESIS_ASSIGNMENT_OPTIONS STYLE WYSIWYG }] } {
		print_msg "No M+II found. Skiping ACF creating/editing"
		return
	}

	# Here's whe1re we set the family.
	set family [get_global_assignment -name FAMILY]
	exec setacf -f$acf_name -sGLOBAL_PROJECT_SYNTHESIS_ASSIGNMENT_OPTIONS DEVICE_FAMILY $family

	set value [get_global_assignment -name NOT_GATE_PUSH_BACK]
	exec setacf -f$acf_name -sDEFINE_LOGIC_SYNTHESIS_STYLE:WYSIWYG.FLEX8000 NOT_GATE_PUSH_BACK $value
	exec setacf -f$acf_name -sDEFINE_LOGIC_SYNTHESIS_STYLE:WYSIWYG.MAX7000 NOT_GATE_PUSH_BACK $value

	set value [get_global_assignment -name DUP_REG_EXTRACTION]
	exec setacf -f$acf_name -sDEFINE_LOGIC_SYNTHESIS_STYLE:WYSIWYG.FLEX8000 REGISTER_OPTIMIZATION $value
	exec setacf -f$acf_name -sDEFINE_LOGIC_SYNTHESIS_STYLE:WYSIWYG.MAX7000 REGISTER_OPTIMIZATION $value

	set value [get_global_assignment -name CARRY_CHAIN]
	exec setacf -f$acf_name -sDEFINE_LOGIC_SYNTHESIS_STYLE:WYSIWYG.FLEX8000 CARRY_CHAIN $value

	set value [get_global_assignment -name CASCADE_CHAIN]
	exec setacf -f$acf_name -sDEFINE_LOGIC_SYNTHESIS_STYLE:WYSIWYG.FLEX8000 CASCADE_CHAIN $value

	exec setacf -f$acf_name -sCOMPILER_INTERFACES_CONFIGURATION EDIF_INPUT_VCC VCC
	exec setacf -f$acf_name -sCOMPILER_INTERFACES_CONFIGURATION EDIF_INPUT_GND GND

	# Note that this one here sets the LMF file.  We need to find out where it is.
	# Assume file is in \bin directory
	set lmf_name $max2synw::max2synw_path
	append lmf_name "max2syn.lmf"
	if { [file exists $lmf_name] } {		
		max2synw::print_msg "Using LMF: $lmf_name"
		exec setacf -f$acf_name -sCOMPILER_INTERFACES_CONFIGURATION EDIF_INPUT_LMF1 "$lmf_name"
	} else {
		max2synw::print_msg "LMF not found: $lmf_name"
	}

	exec setacf -f$acf_name -sCOMPILER_INTERFACES_CONFIGURATION EDIF_INPUT_USE_LMF1 ON
	exec setacf -f$acf_name -sCOMPILER_INTERFACES_CONFIGURATION VERILOG_NETLIST_WRITER OFF
	exec setacf -f$acf_name -sCOMPILER_INTERFACES_CONFIGURATION VHDL_NETLIST_WRITER OFF
	exec setacf -f$acf_name -sCOMPILER_INTERFACES_CONFIGURATION EDIF_NETLIST_WRITER OFF

}

############Predefined flows#############################################

# ----------------------------------------------------------------
#
proc max2synw::on_start_max2syn { } {
#
# Description:	Starts max2syn tool
#
# ----------------------------------------------------------------

	global tool_list
	global user_name
	variable error_count
	variable project_name

	if { ![is_project_open] } {
		max2synw::print_error "No project is open"
		return
	}

	max2synw::print_msg "Start max2syn"
	set max2synw::error_count 0

	max2synw::launch_tool qsyn

	if {$max2synw::error_count == 0} {
		
		# Propagate ACF settings
		max2synw::set_acf
	}

	max2synw::on_process_finished $user_name(qsyn)
}

# ----------------------------------------------------------------
#
proc max2synw::on_start_maxplus2 { } {
#
# Description:	Starts max2syn tool
#
# ----------------------------------------------------------------

	global tool_list
	global user_name
	variable error_count
	variable project_name

	if { ![is_project_open] } {
		max2synw::print_error "No project is open"
		return
	}

	if { ![file exists ${max2synw::project_name}.edf] } {
		max2synw::print_error "${max2synw::project_name}.edf not found"
		max2synw::set_compiler_status waiting
		return
	}

	max2synw::print_msg "Start maxplus2"
	set max2synw::error_count 0

	max2synw::launch_tool mpii

	max2synw::on_process_finished $user_name(mpii)
}

# ----------------------------------------------------------------
#
proc max2synw::on_start_full_compile { } {
#
# Description:	Starts predefined flow i.e. full compilation
#
# ----------------------------------------------------------------

	global tool_list
	variable main_frame
	variable error_count
	variable project_name

	if { ![is_project_open] } {
		max2synw::print_error "No project is open"
		return
	}

	max2synw::print_msg "Start Full Compilation"
	set max2synw::error_count 0

	max2synw::launch_tool qsyn
	if {$max2synw::error_count == 0} {
		
		# Propagate ACF settings
		max2synw::set_acf

		# Now execute maxplus2
		max2synw::launch_tool mpii
	}

	max2synw::on_process_finished "Full compilation"
}

# ----------------------------------------------------------------
#
proc max2synw::on_process_finished { name } {
#
# Description:	Give DLG box indicating success/unsuccess
#
# ----------------------------------------------------------------

	global error_count

	variable app_name
	
	if {$max2synw::error_count == -1} {
		tk_messageBox -type ok -message "$name was stopped" -title "$max2synw::app_name" -icon info
	} elseif {$max2synw::error_count == 0} {
		tk_messageBox -type ok -message "$name was successful" -title "$max2synw::app_name" -icon info
	} else {
		tk_messageBox -type ok -message "$name unsuccessful" -title "$max2synw::app_name" -icon error
	}
}	

# ----------------------------------------------------------------
#
proc max2synw::on_open_max2win { } {
#
# Description:	Starts predefined flow i.e. full compilation
#
# ----------------------------------------------------------------

	global exe_name
	variable project_name

	if { ![is_project_open] } {
		max2synw::print_error "No project is open"
		return
	}

	eval exec $exe_name(mpiiw) $max2synw::project_name &
}	

#############################################################################
## Procedure: on_advance_opt_dlg
##
## Arguments: None
##		
## Description: Opens the option dialog box for advance Settings
##
#############################################################################
proc max2synw::on_advance_opt_dlg { } {

	variable acf

	if { ![is_project_open] } {
		max2synw::print_error "No project is open"
		return
	}

	# create options dialog box

	set dlg [Dialog .adv_synthesis_dlg 	-parent . \
					-modal local \
			        -separator 1 \
			        -title "Advanced Synthesis Settings" \
			        -side  bottom \
			        -anchor c  \
			        -default 0 \
					-cancel 1]

	$dlg add -name ok
	$dlg add -name cancel

	set asgn_list {\
			{"Auto RAM Replacement" AUTO_RAM_RECOGNITION} \
			{"Auto Parallel Expanders" AUTO_PEXP} \
			{"Remove Duplicate Registers" DUP_REG_EXTRACTION} \
			{"Remove Redundant Logic Cells" REMOVE_REDUNDANT_USER_CELLS} \
			{"Ignore LCELL Buffers" IGNORE_LCELL} \
			{"Power-Up Don't Care" ALLOW_POWER_UP_DONT_CARE} \
			{"NOT Gate Push-Back" NOT_GATE_PUSH_BACK} \
	}

	foreach asgn_pair $asgn_list {
		set asgn_name [lindex $asgn_pair 1]
		set max2synw::acf($asgn_name) [get_global_assignment -name $asgn_name]
	}

	set sub_frame_2 [frame $dlg.sub_frame_2 -borderwidth 1 -relief groove -width 15]

	set i 0
	foreach asgn_pair $asgn_list {
		set asgn_user_name [lindex $asgn_pair 0]
		set asgn_name [lindex $asgn_pair 1]
		set chk [checkbutton $sub_frame_2.chk$i -text $asgn_user_name \
				-font {helvetica 10} -width 30 -anchor w \
                -variable max2synw::acf($asgn_name) \
                -onvalue  ON -offvalue OFF]

		grid $chk -padx 7 -pady 4 -sticky w
		incr i
	}

	pack $sub_frame_2 -padx 9 -pady 16 -side top
		
	set cancel [ $dlg draw ]		

	if { !$cancel } {
		
		foreach asgn_pair $asgn_list {
			set asgn_name [lindex $asgn_pair 1]
			set_global_assignment -name $asgn_name $max2synw::acf($asgn_name)
		}

		export_assignment_files
	}

	destroy $dlg
}

#############################################################################
## Procedure: open_qsyn_opt_dlg
##
## Arguments: None
##		
## Description: Opens the option dialog box for 'Technology Mapper'
##
#############################################################################
proc max2synw::open_qsyn_opt_dlg { } {

	variable acf

	if { ![is_project_open] } {
		max2synw::print_error "No project is open"
		return
	}

	# create options dialog box

	set dlg [Dialog .synthesis_dlg 	-parent . \
					-modal local \
			        -separator 1 \
			        -title "Family and Synthesis Settings" \
			        -side  bottom \
			        -anchor c  \
			        -default 0 \
					-cancel 1]

	$dlg add -name ok
	$dlg add -name cancel
	$dlg add -name advance -text "Advanced..." -command "max2synw::on_advance_opt_dlg"

	set max2synw::acf(FAMILY) [get_global_assignment -name FAMILY]
	set max2synw::acf(ROOT) [get_global_assignment -name ROOT]
	if { [string match $max2synw::acf(ROOT) ""] } {
		# Default to Project Name
		set max2synw::acf(ROOT) "|$max2synw::project_name"
	}

	set sub_frame_1 [frame $dlg.sub_frame_1 -borderwidth 1 -relief groove]

	set label_frame_c0 [LabelFrame $sub_frame_1.family_frame -text "Device Family:" -width 23 -font {helvetica 10}]
	set combo_0 [ComboBox $label_frame_c0.combo_1 -text  $max2synw::acf(FAMILY) \
					    -textvariable max2synw::acf(FAMILY) \
						-values [get_family_list] \
						-helptext "Specifies the device family to use for compilation." \
						-width 20 \
						-height 12 \
						-font {helvetica 10} ]
	pack $combo_0 -padx 7 -pady 4

	set label_frame_e1 [LabelFrame $sub_frame_1.root_frame -text "Root Entity Name:" -width 23 -font {helvetica 10}]
	set entry_1 [Entry $label_frame_e1.entry_1 -text  $max2synw::acf(ROOT) \
					    -textvariable max2synw::acf(ROOT) \
						-helptext "Specifies the full hierarchichal path of the root entity." \
						-width 22 \
						-font {helvetica 10} ]
	pack $entry_1 -padx 7 -pady 4

	pack $label_frame_e1 -padx 7 -pady 4
	pack $label_frame_c0 -padx 7 -pady 4

	set asgn_list { \
			{"Verilog Version:" VERILOG_INPUT_VERSION "VERILOG_1995 VERILOG_2001" "Processes Verilog Design File(s) according to the Verilog 2001 or 1995 standard (IEEE Std 1364-2001 or 1364-1995)."} \
			{"VHDL Version:" VHDL_INPUT_VERSION "VHDL87 VHDL93" "Processes VHDL Design File(s) according to the VHDL 1987 or 1993 standard (IEEE Std 1076-1987 or 1076-1993)."}\
			{"Optimization Technique:" OPTIMIZATION_TECHNIQUE "AREA SPEED" "Specifies whether to attempt to achieve maximum speed performance or minimum area usage."} \
			{"Carry Chains:" CARRY_CHAIN "IGNORE AUTO MANUAL" "Specifies how to treat carry chains"} \
			{"Cascade Chains:" CASCADE_CHAIN "IGNORE AUTO MANUAL" "Specifies how to treat carry chains"} \
	}

	foreach asgn_pair $asgn_list {
		set asgn_name [lindex $asgn_pair 1]
		set max2synw::acf($asgn_name) [get_global_assignment -name $asgn_name]
	}

	set i 0
	foreach asgn $asgn_list {

		set asgn_user_name [lindex $asgn 0]
		set asgn_name [lindex $asgn 1]
		set asgn_enums [lindex $asgn 2]
		set asgn_desc [lindex $asgn 3]

		set label_frame [LabelFrame $sub_frame_1.label_frame_$i -text $asgn_user_name -width 23 -font {helvetica 10} ]
		set combo [ComboBox $label_frame.combo_$i -text  $max2synw::acf($asgn_name) \
					    -textvariable max2synw::acf($asgn_name) \
						-height [llength $asgn_enums] \
						-values $asgn_enums \
						-helptext $asgn_desc \
						-width 20 \
						-font {helvetica 10} ]

		pack $combo -padx 7 -pady 4
		pack $label_frame -padx 7 -pady 4
		
		incr i
	}

	pack $sub_frame_1 -padx 9 -pady 2 
	
	set cancel [ $dlg draw ]		

	if { !$cancel } {
		
		set_global_assignment -name FAMILY $max2synw::acf(FAMILY)

		if { [string match $max2synw::acf(ROOT) ""] } {
			# Default to Project Name
			set max2synw::acf(ROOT) "|$max2synw::project_name"
		}
		set_global_assignment -name ROOT $max2synw::acf(ROOT)

		foreach asgn_pair $asgn_list {
			set asgn_name [lindex $asgn_pair 1]
			if { ![string equal $max2synw::acf($asgn_name) ""] } {
				set_global_assignment -name $asgn_name $max2synw::acf($asgn_name)
			}
		}

		export_assignment_files
	}

	destroy $dlg
}

#############################################################################
## Procedure: on_settings_dlg
##
## Arguments: None
##		
## Description: Opens dialog with buttons to other settings
##
#############################################################################
proc max2synw::on_settings_dlg { } {

	variable max2synw_path

	if { ![is_project_open] } {
		max2synw::print_error "No project is open"
		return
	}

	max2synw::open_qsyn_opt_dlg
}

# ----------------------------------------------------------------
#
proc max2synw::on_help {} {
#
# Description:	Display the readme
#				
#
# ----------------------------------------------------------------

	variable version
	variable app_name
	variable max2synw_path

	set exit_help 0

	if [catch {open ${max2synw_path}readme.txt r} ascii_file] {
		max2synw::print_error "Error: Cannot open ${max2synw_path}readme.txt"
	} else {

		toplevel .help
		wm title .help "$max2synw::app_name Readme" 

		set fhelp [frame .help.message -borderwidth 5]
		text $fhelp.text -relief raised -yscrollcommand "$fhelp.scroll set" -height 40 -width 100 -font {Courier 10}
		scrollbar $fhelp.scroll -command "$fhelp.text yview"
		button $fhelp.exit -relief raised -text Exit -default active -command {set exit_help 1}
		pack $fhelp.exit -side bottom -fill x
		pack $fhelp.scroll -side right -fill y
		pack $fhelp.text -side left
		pack $fhelp -side top

		# If no editor is specified, read and display in Log window
		while {([gets $ascii_file ascii_line] >= 0)} {

			$fhelp.text  insert end "$ascii_line\n" infotag
		}	
		close $ascii_file

		vwait exit_help

		destroy .help
	}
}


# ----------------------------------------------------------------
#
proc max2synw::on_license {} {
#
# Description:	Display the max2syn
#				
#
# ----------------------------------------------------------------

	variable version
	variable app_name
	variable max2synw_path

	set exit_license 0

	if [catch {open ${max2synw_path}license.txt r} ascii_file] {
		max2synw::print_error "Error: Cannot open ${max2synw_path}license.txt"
	} else {

		toplevel .license
		wm title .license "$max2synw::app_name License Agreement" 

		set flicense [frame .license.message -borderwidth 5]
		text $flicense.text -relief raised -yscrollcommand "$flicense.scroll set" -height 40 -width 100 -font {Courier 10}
		scrollbar $flicense.scroll -command "$flicense.text yview"
		button $flicense.exit -relief raised -text Exit -default active -command {set exit_license 1}
		pack $flicense.exit -side bottom -fill x
		pack $flicense.scroll -side right -fill y
		pack $flicense.text -side left
		pack $flicense -side top

		# If no editor is specified, read and display in Log window
		while {([gets $ascii_file ascii_line] >= 0)} {

			$flicense.text  insert end "$ascii_line\n" infotag
		}	
		close $ascii_file

		vwait exit_license

		destroy .license
	}
}

# ----------------------------------------------------------------
#
proc max2synw::about_dlg { } {
#
# Description:	"About" Dialog Box
#
# ----------------------------------------------------------------

	variable version
	variable app_name

	set dlg [Dialog .about 	-parent . \
				-modal local \
		        -separator 1 \
                -title   "About MAX+PLUS II Advanced Synthesis" \
		        -side bottom    \
		        -anchor c \
		        -default 0]
	$dlg add -name ok

	set lab1  [Label $dlg.lab1 -text "$app_name $version"]
	set lab2  [Label $dlg.lab2 -text "Copyright (c) 1988-2003 Altera Corporation.  All rights reserved."]
	set lab3  [Label $dlg.lab3 -text "MAX+PLUS is a registered trademark of Altera Corporation in the US"]
	set lab4  [Label $dlg.lab4 -text "and other countries.  Portions of the MAX+PLUS software code and"]
	set lab5  [Label $dlg.lab5 -text "other portions of the code included in this download or on this CD,"]
	set lab6  [Label $dlg.lab6 -text "are licensed to Altera Corporation and are the copyrighted property"]
	set lab7  [Label $dlg.lab7 -text "of third parties who may include, without limitation, Sun"]
	set lab8  [Label $dlg.lab8 -text "Microsystems, The Regents of the University of California, Compass"]
	set lab9  [Label $dlg.lab9 -text "Design Automation, Inc., and Verific Design Automation Inc."]
	set lab10  [Label $dlg.lab10 -text ""]
	set lab11  [Label $dlg.lab11 -text "Warning: This computer program is protected by copyright law and"]
	set lab12  [Label $dlg.lab12 -text "international treaties.  Unauthorized reproduction or distribution of"]
	set lab13  [Label $dlg.lab13 -text "this program, or any portion of it, may result in severe civil and"]
	set lab14  [Label $dlg.lab14 -text "criminal penalties, and will be prosecuted to the maximum extent"]
	set lab15  [Label $dlg.lab15 -text "possible under the law."]

	pack $lab1 -pady 7 -padx 7 -anchor w

	pack $lab2 -pady 0 -padx 7 -anchor w
	pack $lab3 -pady 0 -padx 7 -anchor w
	pack $lab4 -pady 0 -padx 7 -anchor w
	pack $lab5 -pady 0 -padx 7 -anchor w
	pack $lab6 -pady 0 -padx 7 -anchor w
	pack $lab7 -pady 0 -padx 7 -anchor w
	pack $lab8 -pady 0 -padx 7 -anchor w
	pack $lab9 -pady 0 -padx 7 -anchor w

	pack $lab10 -pady 2 -padx 7 -anchor w

	pack $lab11 -pady 0 -padx 7 -anchor w
	pack $lab12 -pady 0 -padx 7 -anchor w
	pack $lab13 -pady 0 -padx 7 -anchor w
	pack $lab14 -pady 0 -padx 7 -anchor w
	pack $lab15 -pady 0 -padx 7 -anchor w


	$dlg draw
	destroy $dlg
}


# Append BWidget directory in auto_path
lappend auto_path max2syn(binpath)
package require BWidget

# Invoke main proc to start the application
max2synw::main

#Set the display size and position of PMA window
set width 1015
set height 675
set x 0
set y 0
wm geom . ${width}x${height}+${x}+${y}

vwait exit_script

