<#
.SYNOPSIS
    When the starting of WinRM fails, it is required to run "WinRM QuickConfig". This script will execute that command on a remote computer using PSExec.

.DESCRIPTION
    Note: Run inside of an administrator PowerShell window
    1. Forces the copy of psexec64.exe to the machine running the script
    2. Opens a powershell shell on the remote computer with PSExec, executing "Set-WSManQuickConfig -Force"
    
.PARAMETER ComputerName
    The string name of the computer you wish to start WinRM on

.NOTES
    Author:                 PolishedCheese
    Date:                   January 9, 2020  
#>

function Repair-WinRM
{
    [CmdletBinding()]
    param (
        [Parameter(Mandatory=$true)]
        [string]$ComputerName
    )
    # Variables that might need to be changed
    $PSExecSource = "\\remoteserver\PsExec64.exe"
    $PSExecHostPath = "C:\Windows\System32\"

    # For flow logic, if it's false, it won't attempt running Set-WSManQuickConfig
    $QuickConfig = $false

    try
    {
        Write-Host -ForegroundColor Yellow "`nTesting WinRM on $ComputerName`n"
        $Success = Test-WSMan -ComputerName $ComputerName -Authentication Default -ErrorAction Stop
        if ($Success)
        {
            Write-Host -ForegroundColor Green "`nWinRM appears to be working on $Computername`n"
        }
    }
    catch
    {
        Write-Warning "WinRM service is either not running or not initialized..."
        $QuickConfig = $true
    }


    # If that didn't work, run quickconfig
    if ($QuickConfig)
    {
        write-host "`n`nRunning WinRM QuickConfig`n"

        # Copy PSExec to the computer running this script
        try
        {
            if (!(Test-Path $PSExecHostPath/PsExec64.exe))
            {
                Write-Host -ForegroundColor Yellow "Copying PsExec to $PsExecHostPath"
                Copy-Item -Path $PSExecSource -Destination $PSExecHostPath
            }
        }

        catch
        {
            Write-Error $_.message
            Write-Warning -Message "Ensure that you have access to $PSExecSource and the PSExec64.exe is located there`n"
        }

        # Execute Set-WSManQuickConfig via PSExec
        try
        {
            $PSExecParams = @{
                ScriptBlock = { Param($ComputerName) C:\Windows\System32\PsExec64.exe \\$ComputerName -d -s -nobanner powershell.exe "& {Set-WSManQuickConfig -Force};"}
                ArgumentList = $Computername
            }    
            Write-Host @PSExecParams
            Invoke-Command @PSExecParams -ErrorAction Stop
        }

        catch
        {
            Write-Error -Message "`nFailed to run Set-WSManQuickConfig remotely."
        }

        finally 
        {
            # Testing WinRM again
            Write-Host -ForegroundColor Yellow "`nConfirming if WinRM is working on $ComputerName"
            $Success = Test-WSMan -ComputerName $ComputerName -Authentication Default -ErrorAction Stop

            if ($Success)
            {
                Write-Host -ForegroundColor Green "`nWinRM appears to be working on $Computername"
            }
            else 
            {
                Write-Warning "`nWinRM still doesn't appear to be working on $Computername. Try running 'winrm quickconfig' locally."
            }

            # Always remove psexec for security's sake. 
            Remove-Item -Path "\\$ComputerName\C$\Windows\System32\PsExec64.exe"
        }
    }
}
