<#
.SYNOPSIS
    Displays the installed software of a computer.

.DESCRIPTION
    1. Tests the connection to the remote computer
    2. Tests to see if WinRM and RemoteRegistry are Running, if they're not, start them.
    3. Collects the contents of the HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\ and HKLM:\SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall\
       registry keys.
    4. Maps the keys to an object then fills an array with the objects
    5. Returns the array of objects
    
.PARAMETER ComputerName
    The string hostname of the computer that we are collecting the list of installed software from.

.EXAMPLE
    "Get-InstalledSoftware -ComputerName Cameron" will collect and display the installed software of "cameron"

.EXAMPLE
    "Get-InstalledSoftware" will collect and display the installed software of $env:COMPUTERNAME

.EXAMPLE
    "Get-InstalledSoftware -ComputerName Cameron | Out-GridView" will collect the installed software of "cameron" and display it via Out-GridView 

.EXAMPLE
    "Get-InstalledSoftware -ComputerName Cameron | Export-CSV -path '.\software.csv' " will collect the installed software of "cameron" and export it to a csv named "software.csv"

.NOTES
    Author:                 Cameron Ratchford
    Date:                   October 21, 2019    

#>


function Get-InstalledSoftware
{
    [CmdletBinding()]
    param (
        [ValidateNotNullOrEmpty()]
        [string]$ComputerName = $env:COMPUTERNAME
    )


    # Test connection to computername
    if (!(Test-Connection -ComputerName $ComputerName -Count 2))
    {
        Write-Warning -Message "Cound not ping $ComputerName"
        Break
    }

    # Attempt to start WinRM if it isn't running
    $TargetWimRM = get-service -Name WinRM -ComputerName $ComputerName -ErrorAction Stop
    if ($TargetWimRM.Status -eq "Stopped")
    {
        try 
        {
            Repair-WinRM -ComputerName $ComputerName
        }
        catch 
        {
            Write-Error $_
            break
        }
    }

    # Start remote registry if it isn't running.
    # Test RemoteRegistry Service
    $TargetRemoteReg = get-service -Name RemoteRegistry -ComputerName $ComputerName
    if ($TargetRemoteReg.Status -match "Stopped")
    {
        try
        {
            start-service $TargetRemoteReg
        }
        catch
        {
            Write-Error $_
            break
        }
    }

    # Custom PS object to store installed programs on
    $Programs = [PSCustomObject]@{
        Name = 'Value'
        DateInstalled = 'Value'
        Publisher = 'Value'
        Version = 'Value'
        EstimatedSize = 'Value'
        Bitness = 'Value'
    }

    # array to store objects in
    $ProgramsList = @()

    # Contents of registry keys for both bitnesses
    # 32 bit
    try
    {
        $List86 = invoke-command -ComputerName $ComputerName -scriptblock {Get-ItemProperty HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\* -Exclude "{*"  | select-object DisplayName, InstallDate, DisplayVersion, EstimatedSize, Publisher} 
    }
    catch
    {
        try 
        {
            write-host "`n"
            Write-Warning "WinRM Could not be started. Running Repair-WinRM...`n"
            Repair-WinRM -ComputerName $ComputerName
        }
        catch 
        {
            Write-Error $_
            break
        }
    }

    # 64 bit
    try
    {
        $List64 = invoke-command -ComputerName $ComputerName -scriptblock {Get-ItemProperty HKLM:\SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall\* -Exclude "{*"  | select-object DisplayName, InstallDate, DisplayVersion, EstimatedSize, Publisher} 
    }
    catch
    {
        try 
        {
            Write-Warning "`nWinRM Could not be started. Running Repair-WinRM...`n`n"
            Repair-WinRM -ComputerName $ComputerName
        }
        catch 
        {
            Write-Error $_
            break
        }
    }     

    # Add registry keys to objects, objects to array
    foreach ($Item in $List86)
    {
        $Programs = [PSCustomObject]@{
            Name = $Item.DisplayName
            Publisher = $Item.Publisher
            DateInstalled = $Item.InstallDate
            Version = $Item.DisplayVersion
            EstimatedSize = $Item.EstimatedSize
            Bitness = 'x86'
        }
        
        if ($ProgramsList.Name -notcontains $Programs.Name)
        {
            if ($null -ne $Programs.Name)
            {
                $ProgramsList += $Programs
            }
        }
    }

    foreach ($Item in $List64)
    {
        $Programs = [PSCustomObject]@{
            Name = $Item.DisplayName
            Publisher = $Item.Publisher
            DateInstalled = $Item.InstallDate
            Version = $Item.DisplayVersion
            EstimatedSize = $Item.EstimatedSize
            Bitness = 'x64'
        }
        
        if ($ProgramsList.Name -notcontains $Programs.Name)
        {
            if ($null -ne $Programs.Name)
            {
                $ProgramsList += $Programs
            }
        }
    }
    
    return $ProgramsList
}
