<#
.SYNOPSIS
    Displays the difference in installed software from 2 computers.
    Extends Get-InstalledSoftware.ps1

.DESCRIPTION
    1. Maps the output of Get-InstalledSoftware computername1 and Get-InstalledSoftware computername2 to variables
    2. Executes a compare, using computername1 as the ReferenceObject and computername2 as the DifferenceObject
    3. Returns the output of compare-object where the values represent the software only present on ComputerName1
    
.PARAMETER ComputerName1
    The string hostname of the computer that we are collecting the list of installed software from to be compared to ComputerName2

.PARAMETER ComputerName2
The string hostname of the computer that we are collecting the list of installed software from to be compared to ComputerName1

.EXAMPLE
    "Compare-InstalledSoftware -ComputerName1 PC-1 -ComputerName2 PC-2" will display the software that is on PC-1 but not PC-2

.EXAMPLE
    "Compare-InstalledSoftware -ComputerName1 PC-1 -ComputerName2 PC-2 | Out-GridView" will display the software that is on PC-1 but not PC-2 via Out-GridView 

.EXAMPLE
    "Compare-InstalledSoftware -ComputerName1 PC-1 -ComputerName2 PC-2 | Export-CSV -path '.\diff.csv' " will export the software that is on PC-1 but not PC-2 to a csv named "diff.csv"

.NOTES
    Author:                 Cameron Ratchford
    Date:                   November 27, 2019
#>

function Compare-InstalledSoftware
{
    [CmdletBinding()]
    param (
        [Parameter(Mandatory=$true)]
        [ValidateNotNullOrEmpty()]
        [string]$ComputerName1,

        [Parameter(Mandatory=$true)]
        [ValidateNotNullOrEmpty()]
        [string]$ComputerName2
    )

    $SoftwareList1 = get-installedsoftware -ComputerName $ComputerName1
    $SoftwareList2 = get-installedsoftware -ComputerName $ComputerName2

    $diffs = $SoftwareList1 | Where-Object -FilterScript {$SoftwareList2.Name -notcontains $_.Name}

    return $diffs
}
